// Copyright (C) 2025 Vaadin Ltd
// This program is available under Vaadin Commercial License and Service Terms.
// See <https://vaadin.com/commercial-license-and-service-terms> for the full license.
package com.vaadin.controlcenter.starter;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.context.event.EventListener;
import org.springframework.security.core.context.SecurityContextHolderStrategy;
import org.springframework.security.oauth2.client.OAuth2AuthorizeRequest;
import org.springframework.security.oauth2.client.OAuth2AuthorizedClientManager;
import org.springframework.security.oauth2.client.OAuth2AuthorizedClientService;
import org.springframework.security.oauth2.client.authentication.OAuth2AuthenticationToken;

import com.vaadin.flow.component.HeartbeatEvent;
import com.vaadin.flow.component.HeartbeatListener;
import com.vaadin.flow.server.ServiceInitEvent;
import com.vaadin.flow.spring.security.AuthenticationContext;

/// A UI heartbeat listener to keep the Keycloak Session alive by re-authorizing the client at every UI heartbeat.
class KeycloakSessionRefreshHeartbeatListener implements HeartbeatListener {

    private static final Logger LOGGER = LoggerFactory.getLogger(KeycloakSessionRefreshHeartbeatListener.class);

    private final transient SecurityContextHolderStrategy securityContextHolderStrategy;

    private final transient OAuth2AuthorizedClientService clientService;

    private final transient OAuth2AuthorizedClientManager clientManager;

    private final transient AuthenticationContext authenticationContext;

    KeycloakSessionRefreshHeartbeatListener(
            SecurityContextHolderStrategy securityContextHolderStrategy,
            OAuth2AuthorizedClientService clientService,
            OAuth2AuthorizedClientManager clientManager,
            AuthenticationContext authenticationContext) {
        this.securityContextHolderStrategy = securityContextHolderStrategy;
        this.clientService = clientService;
        this.clientManager = clientManager;
        this.authenticationContext = authenticationContext;
    }

    @EventListener
    void serviceInit(ServiceInitEvent serviceInitEvent) {
        LOGGER.debug("Service initialized, registering heartbeat listener");
        serviceInitEvent
                .getSource()
                .addUIInitListener(uiInitEvent -> uiInitEvent.getUI().addHeartbeatListener(this));
    }

    @Override
    public void heartbeat(HeartbeatEvent event) {
        LOGGER.debug("Heartbeat received, attempting to refresh OIDC Session");
        var auth = securityContextHolderStrategy.getContext().getAuthentication();
        if (auth instanceof OAuth2AuthenticationToken token) {
            var registrationId = token.getAuthorizedClientRegistrationId();
            var client = clientService.loadAuthorizedClient(registrationId, token.getName());
            if (client == null) {
                LOGGER.warn("Could not find authorized client for registrationId {}, forcing logout", registrationId);
                authenticationContext.logout();
                return;
            }
            var authorizeRequest = OAuth2AuthorizeRequest.withClientRegistrationId(registrationId)
                    .principal(token)
                    .build();
            var newClient = clientManager.authorize(authorizeRequest);
            if (newClient == null) {
                LOGGER.warn("Refresh attempt failed for registrationId {}, forcing logout", registrationId);
                authenticationContext.logout();
                return;
            }
            LOGGER.debug("Refresh attempt successful for registrationId {}", registrationId);
            clientService.saveAuthorizedClient(newClient, token);
        }
    }
}
