// Copyright (C) 2025 Vaadin Ltd
// This program is available under Vaadin Commercial License and Service Terms.
// See <https://vaadin.com/commercial-license-and-service-terms> for the full license.
package com.vaadin.controlcenter.starter;

import java.util.List;
import java.util.Map;

import com.fasterxml.jackson.databind.ObjectMapper;
import io.fabric8.kubernetes.api.model.Pod;
import io.fabric8.kubernetes.client.KubernetesClient;
import org.springframework.aot.hint.annotation.RegisterReflectionForBinding;
import org.springframework.boot.autoconfigure.AutoConfiguration;
import org.springframework.boot.autoconfigure.condition.ConditionalOnCloudPlatform;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.boot.availability.ApplicationAvailability;
import org.springframework.boot.cloud.CloudPlatform;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.cloud.kubernetes.commons.PodUtils;
import org.springframework.context.ApplicationContext;
import org.springframework.context.annotation.Bean;
import org.springframework.core.Ordered;
import org.springframework.core.ParameterizedTypeReference;
import org.springframework.core.annotation.Order;
import org.springframework.security.config.annotation.web.builders.HttpSecurity;
import org.springframework.security.config.annotation.web.configurers.oauth2.client.OidcBackChannelLogoutHandler;
import org.springframework.security.core.context.SecurityContextHolderStrategy;
import org.springframework.security.oauth2.client.OAuth2AuthorizedClientManager;
import org.springframework.security.oauth2.client.OAuth2AuthorizedClientService;
import org.springframework.security.oauth2.client.oidc.session.InMemoryOidcSessionRegistry;
import org.springframework.security.oauth2.client.oidc.session.OidcSessionRegistry;
import org.springframework.security.oauth2.client.oidc.userinfo.OidcUserService;
import org.springframework.security.oauth2.client.registration.ClientRegistrationRepository;
import org.springframework.security.oauth2.client.registration.ClientRegistrations;
import org.springframework.security.oauth2.client.registration.InMemoryClientRegistrationRepository;
import org.springframework.security.oauth2.client.registration.SupplierClientRegistrationRepository;
import org.springframework.security.web.SecurityFilterChain;
import org.springframework.util.Assert;
import org.springframework.web.client.RestClient;

import com.vaadin.flow.spring.security.AuthenticationContext;
import com.vaadin.flow.spring.security.VaadinAwareSecurityContextHolderStrategy;

import static org.springframework.boot.actuate.autoconfigure.security.servlet.EndpointRequest.toAnyEndpoint;

/// Configuration for Control Center.
@AutoConfiguration
@EnableConfigurationProperties(ControlCenterProperties.class)
public class ControlCenterConfiguration {

    static final ParameterizedTypeReference<Map<String, Object>> METADATA_TYPE = new ParameterizedTypeReference<>() {};

    static final ObjectMapper MAPPER = new ObjectMapper();

    private final ControlCenterProperties properties;

    private final RestClient restClient;

    ControlCenterConfiguration(ControlCenterProperties properties, RestClient.Builder restClientBuilder) {
        this.properties = properties;
        this.restClient = restClientBuilder.build();
    }

    @Bean
    KeycloakSessionRefreshHeartbeatListener oidcRefreshHeartbeatListener(
            SecurityContextHolderStrategy securityContextHolderStrategy,
            OAuth2AuthorizedClientService clientService,
            OAuth2AuthorizedClientManager clientManager,
            AuthenticationContext authenticationContext) {
        return new KeycloakSessionRefreshHeartbeatListener(
                securityContextHolderStrategy, clientService, clientManager, authenticationContext);
    }

    @Bean
    @ConditionalOnMissingBean
    SecurityContextHolderStrategy securityContextHolderStrategy() {
        return new VaadinAwareSecurityContextHolderStrategy();
    }

    @Bean
    @ConditionalOnMissingBean
    OidcSessionRegistry oidcSessionRegistry() {
        return new InMemoryOidcSessionRegistry();
    }

    @Bean
    @ConditionalOnCloudPlatform(CloudPlatform.KUBERNETES)
    KubernetesBackChannelLogoutHandler backChannelLogoutHandler(
            RestClient.Builder restClientBuilder,
            KubernetesClient kubernetesClient,
            PodUtils<Pod> podUtils,
            OidcSessionRegistry sessionRegistry) {
        var delegate = new OidcBackChannelLogoutHandler(sessionRegistry);
        return new KubernetesBackChannelLogoutHandler(restClientBuilder, kubernetesClient, podUtils, delegate);
    }

    @Bean
    @ConditionalOnCloudPlatform(CloudPlatform.KUBERNETES)
    ReadinessStateSignalHandler readinessStateSignalHandler(ApplicationContext applicationContext) {
        return new ReadinessStateSignalHandler(applicationContext);
    }

    @Bean
    @ConditionalOnCloudPlatform(CloudPlatform.KUBERNETES)
    StartupStateHealthIndicator startupStateHealthIndicator(ApplicationAvailability availability) {
        return new StartupStateHealthIndicator(availability);
    }

    @Bean
    @RegisterReflectionForBinding(VaadinActuatorEndpoint.VaadinInfo.class)
    VaadinActuatorEndpoint vaadinActuatorEndpoint() {
        return new VaadinActuatorEndpoint();
    }

    @Bean
    @ConditionalOnMissingBean
    OidcUserService keycloakUserService() {
        var oidcUserService = new OidcUserService();
        oidcUserService.setOidcUserMapper(new KeycloakUserMapper(MAPPER));
        return oidcUserService;
    }

    @Bean
    @Order(Ordered.HIGHEST_PRECEDENCE)
    SecurityFilterChain actuatorSecurityFilterChain(HttpSecurity http) throws Exception {
        // Authorize any actuator endpoint request, since the management server is not exposed to the outside world.
        return http.securityMatcher(toAnyEndpoint())
                .authorizeHttpRequests(authorize -> authorize.anyRequest().permitAll())
                .build();
    }

    @Bean
    @ConditionalOnMissingBean
    ClientRegistrationRepository clientRegistrationRepository() {
        return new SupplierClientRegistrationRepository(this::lazyClientRegistrationRepository);
    }

    InMemoryClientRegistrationRepository lazyClientRegistrationRepository() {
        var clientId = properties.getClientId();
        var clientSecret = properties.getClientSecret();
        var registrationId = properties.getRegistrationId();
        var issuerBackendUri = properties.getIssuerBackendUri();
        var issuerFrontendUri = properties.getIssuerFrontendUri();
        var wellKnownConfigurationUri = issuerBackendUri + "/.well-known/openid-configuration";
        var metadata =
                restClient.get().uri(wellKnownConfigurationUri).retrieve().body(METADATA_TYPE);
        Assert.notNull(metadata, couldNotRetrieveMetadata(wellKnownConfigurationUri));
        var metadataIssuer = (String) metadata.get("issuer");
        Assert.state(issuerFrontendUri.equals(metadataIssuer), issuerMismatch(metadataIssuer, issuerFrontendUri));
        var registration = ClientRegistrations.fromOidcConfiguration(metadata)
                .clientId(clientId)
                .clientSecret(clientSecret)
                .registrationId(registrationId)
                .scope(properties.getClientScope())
                .build();
        return new InMemoryClientRegistrationRepository(List.of(registration));
    }

    private String couldNotRetrieveMetadata(String issuerUri) {
        return "Could not retrieve metadata from issuer at " + issuerUri;
    }

    private String issuerMismatch(String metadataIssuer, String expectedIssuer) {
        return "The issuer '%s' provided in the configuration metadata did not match the expected issuer '%s'"
                .formatted(metadataIssuer, expectedIssuer);
    }
}
