// Copyright (C) 2025 Vaadin Ltd
// This program is available under Vaadin Commercial License and Service Terms.
// See <https://vaadin.com/commercial-license-and-service-terms> for the full license.
package com.vaadin.controlcenter.starter;

import java.util.concurrent.atomic.AtomicInteger;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.boot.actuate.endpoint.annotation.Endpoint;
import org.springframework.boot.actuate.endpoint.annotation.ReadOperation;
import org.springframework.context.event.EventListener;

import com.vaadin.flow.server.ServiceInitEvent;
import com.vaadin.flow.server.SessionDestroyEvent;
import com.vaadin.flow.server.SessionInitEvent;
import com.vaadin.flow.server.Version;

/// An actuator endpoint that provides information about the Vaadin app.
@Endpoint(id = VaadinActuatorEndpoint.ENDPOINT_ID)
public class VaadinActuatorEndpoint {

    /// The ID of the Vaadin actuator endpoint.
    public static final String ENDPOINT_ID = "vaadin";

    private static final Logger LOGGER = LoggerFactory.getLogger(VaadinActuatorEndpoint.class);

    private final AtomicInteger counter = new AtomicInteger();

    VaadinActuatorEndpoint() {}

    @EventListener
    void serviceInit(ServiceInitEvent serviceInitEvent) {
        LOGGER.debug("Service initialized, registering session listeners");
        var service = serviceInitEvent.getSource();
        service.addSessionInitListener(this::sessionInit);
        service.addSessionDestroyListener(this::sessionDestroy);
    }

    void sessionInit(SessionInitEvent sessionInitEvent) {
        var session = sessionInitEvent.getSession();
        if (LOGGER.isDebugEnabled() && session != null) {
            var sessionId = session.getSession().getId();
            LOGGER.debug("Session initialized: {}", sessionId);
        }
        var count = counter.incrementAndGet();
        LOGGER.debug("Session counter incremented to {}", count);
    }

    void sessionDestroy(SessionDestroyEvent sessionDestroyEvent) {
        var session = sessionDestroyEvent.getSession();
        if (LOGGER.isDebugEnabled() && session != null) {
            var sessionId = session.getSession().getId();
            LOGGER.debug("Session destroyed: {}", sessionId);
        }
        var count = counter.decrementAndGet();
        LOGGER.debug("Session counter decremented to {}", count);
    }

    /// The read operation of this endpoint.
    ///
    /// @return a record with information about the Vaadin app
    @ReadOperation
    public VaadinInfo vaadinInfo() {
        return new VaadinInfo(Version.getFullVersion(), counter.get());
    }

    /// A record with information about the Vaadin app.
    ///
    /// @param version the current version
    /// @param sessions the number of active sessions
    public record VaadinInfo(String version, int sessions) {}
}
