// Copyright (C) 2025 Vaadin Ltd
// This program is available under Vaadin Commercial License and Service Terms.
// See <https://vaadin.com/commercial-license-and-service-terms> for the full license.
package com.vaadin.controlcenter.starter;

import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.function.BiFunction;

import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.ObjectMapper;
import org.springframework.security.core.GrantedAuthority;
import org.springframework.security.oauth2.client.oidc.userinfo.OidcUserRequest;
import org.springframework.security.oauth2.core.oidc.OidcUserInfo;
import org.springframework.security.oauth2.core.oidc.user.DefaultOidcUser;
import org.springframework.security.oauth2.core.oidc.user.OidcUser;
import org.springframework.security.oauth2.core.oidc.user.OidcUserAuthority;
import org.springframework.security.oauth2.jwt.JwtValidators;
import org.springframework.security.oauth2.jwt.NimbusJwtDecoder;
import org.springframework.util.StringUtils;

class KeycloakUserMapper implements BiFunction<OidcUserRequest, OidcUserInfo, OidcUser> {

    static final TypeReference<Map<String, List<String>>> LIST_TYPE = new TypeReference<>() {};

    static final TypeReference<Map<String, Map<String, List<String>>>> MAP_TYPE = new TypeReference<>() {};

    static final String REALM_ACCESS_CLAIM = "realm_access";

    static final String RESOURCE_ACCESS_CLAIM = "resource_access";

    static final String ROLES_CLAIM = "roles";

    static final String ROLE_PREFIX = "ROLE_";

    static final String SCOPE_PREFIX = "SCOPE_";

    private final ObjectMapper mapper;

    KeycloakUserMapper(ObjectMapper mapper) {
        this.mapper = mapper;
    }

    @Override
    public OidcUser apply(OidcUserRequest userRequest, OidcUserInfo userInfo) {
        var authorities = new LinkedHashSet<GrantedAuthority>();
        var accessToken = userRequest.getAccessToken();
        accessToken.getScopes().stream().map(OidcScopeAuthority::new).forEach(authorities::add);
        var clientRegistration = userRequest.getClientRegistration();
        var clientId = clientRegistration.getClientId();
        var providerDetails = clientRegistration.getProviderDetails();
        var issuerUri = providerDetails.getIssuerUri();
        var jwkSetUri = providerDetails.getJwkSetUri();
        var jwtDecoder = NimbusJwtDecoder.withJwkSetUri(jwkSetUri).build();
        jwtDecoder.setJwtValidator(JwtValidators.createDefaultWithIssuer(issuerUri));
        var jwt = jwtDecoder.decode(accessToken.getTokenValue());
        if (jwt.hasClaim(REALM_ACCESS_CLAIM)) {
            var claim = mapper.convertValue(jwt.getClaimAsMap(REALM_ACCESS_CLAIM), LIST_TYPE);
            var roles = claim.getOrDefault(ROLES_CLAIM, List.of());
            roles.stream().map(KeycloakRealmRole::new).forEach(authorities::add);
        }
        if (jwt.hasClaim(RESOURCE_ACCESS_CLAIM)) {
            var claim = mapper.convertValue(jwt.getClaimAsMap(RESOURCE_ACCESS_CLAIM), MAP_TYPE);
            var resources = claim.getOrDefault(clientId, Map.of());
            var roles = resources.getOrDefault(ROLES_CLAIM, List.of());
            roles.stream().map(role -> new KeycloakClientRole(clientId, role)).forEach(authorities::add);
        }
        var userNameAttributeName = providerDetails.getUserInfoEndpoint().getUserNameAttributeName();
        if (StringUtils.hasText(userNameAttributeName)) {
            authorities.add(new OidcUserAuthority(userRequest.getIdToken(), userInfo, userNameAttributeName));
            return new DefaultOidcUser(authorities, userRequest.getIdToken(), userNameAttributeName);
        } else {
            authorities.add(new OidcUserAuthority(userRequest.getIdToken(), userInfo));
            return new DefaultOidcUser(authorities, userRequest.getIdToken(), userInfo);
        }
    }

    record OidcScopeAuthority(String scope) implements GrantedAuthority {

        @Override
        public String getAuthority() {
            return SCOPE_PREFIX + scope;
        }
    }

    record KeycloakRealmRole(String role) implements GrantedAuthority {

        @Override
        public String getAuthority() {
            return ROLE_PREFIX + role;
        }
    }

    record KeycloakClientRole(String client, String role) implements GrantedAuthority {

        @Override
        public String getAuthority() {
            return ROLE_PREFIX + role;
        }
    }
}
