// Copyright (C) 2025 Vaadin Ltd
// This program is available under Vaadin Commercial License and Service Terms.
// See <https://vaadin.com/commercial-license-and-service-terms> for the full license.
package com.vaadin.controlcenter.starter;

import java.util.Optional;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.context.ApplicationContext;
import org.springframework.security.config.annotation.web.builders.HttpSecurity;
import org.springframework.security.config.annotation.web.configurers.AbstractHttpConfigurer;
import org.springframework.security.config.annotation.web.configurers.oauth2.client.OidcLogoutConfigurer;
import org.springframework.util.Assert;
import org.springframework.util.StringUtils;

import com.vaadin.flow.spring.security.VaadinSecurityConfigurer;

import static java.util.Objects.requireNonNullElseGet;

/// A configurer for [HttpSecurity] builders designed to set up the security filter chain for Control Center.
///
/// Usage example:
/// ~~~java
/// @Configuration
/// @EnableWebSecurity
/// public class MyConfiguration {
///
///     @Bean
///     SecurityFilterChain mySecurityFilterChain(HttpSecurity http) throws Exception {
///         return http.with(new ControlCenterSecurityConfigurer(), Customizer.withDefaults()).build();
///     }
/// }
/// ~~~
public final class ControlCenterSecurityConfigurer
        extends AbstractHttpConfigurer<ControlCenterSecurityConfigurer, HttpSecurity> {

    private static final String MISSING_REGISTRATION_ID =
            """
            The registrationId cannot be empty. When using ControlCenterSecurityConfigurer the registrationId property
            needs to be provided setting vaadin.control-center.registration-id in the application.properties (or .yaml)
            file, or in the app's Kubernetes ConfigMap (or Secret). The property is configured automatically when the
            app is deployed to Kubernetes using with Control Center and spec.keycloak.realm is defined.
            """;

    private static final String MISSING_APPLICATION_CONTEXT =
            """
            No ApplicationContext available. Make sure to add the configurer to the HttpSecurity builder using the
            with(configurer, customizer) method.
            """;

    private static final Logger LOGGER = LoggerFactory.getLogger(ControlCenterSecurityConfigurer.class);

    private ControlCenterSecurityConfigurer() {}

    /// Provides a new instance of [ControlCenterSecurityConfigurer] that can be used to configure security for
    /// applications using Control Center.
    ///
    /// @return a new instance of this configurer
    public static ControlCenterSecurityConfigurer controlCenter() {
        return new ControlCenterSecurityConfigurer();
    }

    @Override
    public void init(HttpSecurity http) throws Exception {
        var registrationId = getProperties().getRegistrationId();
        if (StringUtils.hasText(registrationId)) {
            http.with(getVaadinSecurityConfigurer(), this::customize).oidcLogout(this::customize);
        } else {
            LOGGER.warn("No registrationId provided. Skipping Control Center security configuration.");
        }
    }

    private void customize(VaadinSecurityConfigurer configurer) {
        var registrationId = getProperties().getRegistrationId();
        Assert.hasText(registrationId, MISSING_REGISTRATION_ID);
        configurer.oauth2LoginPage("/oauth2/authorization/" + registrationId);
    }

    private void customize(OidcLogoutConfigurer<HttpSecurity> configurer) {
        var logoutHandler = getSharedObjectOrBean(KubernetesBackChannelLogoutHandler.class);
        if (logoutHandler != null) {
            configurer.backChannel(bc -> bc.logoutHandler(logoutHandler));
        }
    }

    private VaadinSecurityConfigurer getVaadinSecurityConfigurer() {
        var existingConfigurer = getBuilder().getConfigurer(VaadinSecurityConfigurer.class);
        return requireNonNullElseGet(existingConfigurer, VaadinSecurityConfigurer::vaadin);
    }

    private ControlCenterProperties getProperties() {
        return getSharedObjectOrBean(ControlCenterProperties.class);
    }

    private <T> T getSharedObjectOrBean(Class<T> type) {
        var sharedObject = getBuilder().getSharedObject(type);
        return Optional.ofNullable(sharedObject).orElseGet(() -> {
            var applicationContext = getBuilder().getSharedObject(ApplicationContext.class);
            Assert.notNull(applicationContext, MISSING_APPLICATION_CONTEXT);
            return applicationContext.getBeanProvider(type).getIfAvailable();
        });
    }
}
