/*-
 * Copyright (C) 2025 Vaadin Ltd
 *
 * This program is available under Vaadin Commercial License and Service Terms.
 *
 * See <https://vaadin.com/commercial-license-and-service-terms> for the full license.
-*/
package com.vaadin.controlcenter.starter.i18n;

import java.util.Arrays;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.vaadin.flow.component.Text;
import com.vaadin.flow.component.UI;
import com.vaadin.flow.component.button.Button;
import com.vaadin.flow.component.icon.Icon;
import com.vaadin.flow.component.icon.VaadinIcon;
import com.vaadin.flow.component.notification.Notification;
import com.vaadin.flow.component.notification.NotificationVariant;
import com.vaadin.flow.component.orderedlayout.FlexComponent;
import com.vaadin.flow.component.orderedlayout.HorizontalLayout;
import com.vaadin.flow.server.ServiceInitEvent;
import com.vaadin.flow.server.VaadinRequest;
import com.vaadin.flow.server.VaadinServiceInitListener;

/**
 * A service init listener that adds an I18N preview banner to the UI.
 */
class I18NPreviewServiceInitListener implements VaadinServiceInitListener {

    private static final String I18N_PREVIEW_COOKIE = "i18n-preview";

    private final transient Logger logger = LoggerFactory.getLogger(getClass());

    Button moveButton;

    Button exitButton;

    @Override
    public void serviceInit(ServiceInitEvent serviceInitEvent) {

        logger.info("Initializing I18nPreviewService");

        serviceInitEvent.getSource().addUIInitListener(event -> {

            if (!isPreviewEnabled()) {
                return;
            }

            UI ui = event.getUI();
            ui.setLocale(VaadinRequest.getCurrent().getLocale());

            Icon downIcon = VaadinIcon.ARROW_DOWN.create();
            Icon upIcon = VaadinIcon.ARROW_UP.create();
            moveButton = new Button(upIcon);
            moveButton.setAriaLabel("Move notification to top");
            moveButton.getStyle().setMargin("0 0 0 0");

            exitButton = new Button(VaadinIcon.CLOSE.create());
            exitButton.addClickListener(e -> {
                String currentLocation = ui.getInternals()
                        .getActiveViewLocation().getPath();
                String newLocation = currentLocation + "?i18n-preview=disable";
                ui.getPage().setLocation(newLocation);
            });
            exitButton.getStyle().setMargin("0 0 0 0");

            var layout = new HorizontalLayout(
                    new Text("Control Center translation preview"), moveButton,
                    exitButton);
            layout.setAlignItems(FlexComponent.Alignment.CENTER);

            Notification notification = new Notification();
            notification.add(layout);
            notification.addThemeVariants(NotificationVariant.LUMO_WARNING);
            notification.setPosition(Notification.Position.BOTTOM_END);
            moveButton.addClickListener(e -> {
                if (moveButton.getIcon().equals(downIcon)) {
                    notification.setPosition(Notification.Position.BOTTOM_END);
                    moveButton.setIcon(upIcon);
                    moveButton.setAriaLabel("Move notification to top");
                } else {
                    notification.setPosition(Notification.Position.TOP_END);
                    moveButton.setIcon(downIcon);
                    moveButton.setAriaLabel("Move notification to bottom");
                }
            });

            ui.add(notification);
            notification.open();
        });
    }

    private boolean isPreviewEnabled() {
        if (VaadinRequest.getCurrent() == null) {
            return false;
        }
        if (VaadinRequest.getCurrent().getCookies() == null) {
            return false;
        }
        return Arrays.stream(VaadinRequest.getCurrent().getCookies())
                .filter(cookie -> I18N_PREVIEW_COOKIE.equals(cookie.getName()))
                .findFirst()
                .map(cookie -> Boolean.parseBoolean(cookie.getValue()))
                .orElse(false);
    }
}
