/*-
 * Copyright (C) 2024 Vaadin Ltd
 *
 * This program is available under Vaadin Commercial License and Service Terms.
 *
 * See <https://vaadin.com/commercial-license-and-service-terms> for the full license.
-*/
package com.vaadin.controlcenter.starter.idm;

import java.util.Collection;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;

import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.ObjectMapper;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.AutoConfiguration;
import org.springframework.boot.autoconfigure.condition.ConditionalOnCloudPlatform;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.boot.cloud.CloudPlatform;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.cloud.context.config.annotation.RefreshScope;
import org.springframework.context.annotation.Bean;
import org.springframework.security.config.annotation.web.builders.HttpSecurity;
import org.springframework.security.config.annotation.web.configuration.EnableWebSecurity;
import org.springframework.security.config.annotation.web.configuration.WebSecurityCustomizer;
import org.springframework.security.config.annotation.web.configurers.AuthorizeHttpRequestsConfigurer;
import org.springframework.security.config.annotation.web.configurers.CsrfConfigurer;
import org.springframework.security.config.annotation.web.configurers.LogoutConfigurer;
import org.springframework.security.config.annotation.web.configurers.oauth2.client.OAuth2LoginConfigurer;
import org.springframework.security.config.annotation.web.configurers.oauth2.client.OidcLogoutConfigurer;
import org.springframework.security.core.GrantedAuthority;
import org.springframework.security.core.authority.SimpleGrantedAuthority;
import org.springframework.security.oauth2.client.OAuth2AuthorizedClientService;
import org.springframework.security.oauth2.client.oidc.userinfo.OidcUserRequest;
import org.springframework.security.oauth2.client.oidc.userinfo.OidcUserService;
import org.springframework.security.oauth2.client.oidc.web.logout.OidcClientInitiatedLogoutSuccessHandler;
import org.springframework.security.oauth2.client.registration.ClientRegistrationRepository;
import org.springframework.security.oauth2.core.oidc.OidcUserInfo;
import org.springframework.security.oauth2.core.oidc.user.DefaultOidcUser;
import org.springframework.security.oauth2.core.oidc.user.OidcUser;
import org.springframework.security.oauth2.core.oidc.user.OidcUserAuthority;
import org.springframework.security.oauth2.jwt.JwtValidators;
import org.springframework.security.oauth2.jwt.NimbusJwtDecoder;
import org.springframework.security.web.SecurityFilterChain;
import org.springframework.security.web.authentication.AnonymousAuthenticationFilter;
import org.springframework.util.Assert;
import org.springframework.util.StringUtils;

import com.vaadin.flow.spring.security.AuthenticationContext;
import com.vaadin.flow.spring.security.RequestUtil;
import com.vaadin.flow.spring.security.UidlRedirectStrategy;
import com.vaadin.flow.spring.security.VaadinWebSecurity;

/**
 * This configuration bean is provided to autoconfigure the security to allow
 * single sign-on against identity provider used in Control Center.
 */
@AutoConfiguration
@EnableWebSecurity
@ConditionalOnCloudPlatform(CloudPlatform.KUBERNETES)
@ConditionalOnMissingBean(IdentityManagementConfiguration.class)
@EnableConfigurationProperties(IdentityManagementProperties.class)
public class IdentityManagementConfiguration extends VaadinWebSecurity {

    private static final String BACK_CHANNEL_LOGOUT_URI = "{baseScheme}://{baseHost}{basePort}/logout";

    private static final String REALM_ACCESS_CLAIM = "realm_access";

    private static final String RESOURCE_ACCESS_CLAIM = "resource_access";

    private static final String ROLES_CLAIM = "roles";

    private static final String ROLE_PREFIX = "ROLE_";

    private static final String SCOPE_PREFIX = "SCOPE_";

    private static final ObjectMapper MAPPER = new ObjectMapper();

    private IdentityManagementProperties properties;

    private ClientRegistrationRepository clientRegistrationRepository;

    private OAuth2AuthorizedClientService oAuth2AuthorizedClientService;

    private RequestUtil requestUtil;

    @Autowired
    void setProperties(IdentityManagementProperties properties) {
        this.properties = properties;
    }

    @Autowired
    void setClientRegistrationRepository(
            ClientRegistrationRepository clientRegistrationRepository) {
        this.clientRegistrationRepository = clientRegistrationRepository;
    }

    @Autowired
    void setOAuth2AuthorizedClientService(
            OAuth2AuthorizedClientService oAuth2AuthorizedClientService) {
        this.oAuth2AuthorizedClientService = oAuth2AuthorizedClientService;
    }

    @Autowired
    void setRequestUtil(RequestUtil requestUtil) {
        this.requestUtil = requestUtil;
    }

    @Bean(name = "VaadinSecurityFilterChainBean")
    @ConditionalOnMissingBean(name = "VaadinSecurityFilterChainBean")
    @Override
    @RefreshScope
    @SuppressWarnings("java:S6830")
    public SecurityFilterChain filterChain(HttpSecurity http) throws Exception {
        return super.filterChain(http);
    }

    @Bean(name = "VaadinWebSecurityCustomizerBean")
    @ConditionalOnMissingBean(name = "VaadinWebSecurityCustomizerBean")
    @Override
    @RefreshScope
    @SuppressWarnings("java:S6830")
    public WebSecurityCustomizer webSecurityCustomizer() {
        return super.webSecurityCustomizer();
    }

    @Bean(name = "VaadinAuthenticationContext")
    @ConditionalOnMissingBean(name = "VaadinAuthenticationContext")
    @Override
    @RefreshScope
    @SuppressWarnings("java:S6830")
    public AuthenticationContext getAuthenticationContext() {
        return super.getAuthenticationContext();
    }

    @Bean
    @ConditionalOnMissingBean
    OidcUserService oidcUserService() {
        var oidcUserService = new OidcUserService();
        oidcUserService.setOidcUserMapper(this::keycloakUserMapper);
        return oidcUserService;
    }

    @Override
    protected void configure(HttpSecurity http) throws Exception {
        if (properties.isEnabled()) {
            withIdentityManagementEnabled(http);
            super.configure(http);
        } else {
            withIdentityManagementDisabled(http);
        }
    }

    protected void withIdentityManagementEnabled(HttpSecurity http)
            throws Exception {
        http.addFilterBefore(
                new RefreshTokenFilter(oAuth2AuthorizedClientService),
                AnonymousAuthenticationFilter.class);
        http.authorizeHttpRequests(this::requestWhitelist);
        http.oauth2Login(this::configureOidcLogin);
        setOAuth2LoginPage(http, properties.getLoginRoute());
        http.oidcLogout(this::configureOidcLogout);
        http.logout(this::configureLogout);
    }

    protected void withIdentityManagementDisabled(HttpSecurity http)
            throws Exception {
        http.csrf(this::ignoreInternalRequests);
        http.authorizeHttpRequests(this::authorizeAnyRequest);
    }

    protected void requestWhitelist(
            AuthorizeHttpRequestsConfigurer<HttpSecurity>.AuthorizationManagerRequestMatcherRegistry urlRegistry) {
        urlRegistry.requestMatchers("/actuator/**").permitAll();
    }

    protected void ignoreInternalRequests(CsrfConfigurer<HttpSecurity> csrf) {
        csrf.ignoringRequestMatchers(requestUtil::isFrameworkInternalRequest);
    }

    private void configureOidcLogin(OAuth2LoginConfigurer<HttpSecurity> login) {
        login.defaultSuccessUrl(properties.getLoginSuccessRoute());
    }

    private void configureOidcLogout(
            OidcLogoutConfigurer<HttpSecurity> logout) {
        logout.backChannel(
                backChannel -> backChannel.logoutUri(BACK_CHANNEL_LOGOUT_URI));
    }

    private void configureLogout(LogoutConfigurer<HttpSecurity> logout) {
        var logoutSuccessHandler = new OidcClientInitiatedLogoutSuccessHandler(
                clientRegistrationRepository);
        var logoutSuccessRoute = properties.getLogoutSuccessRoute();
        logoutSuccessHandler.setPostLogoutRedirectUri(logoutSuccessRoute);
        logoutSuccessHandler.setRedirectStrategy(new UidlRedirectStrategy());
        logout.logoutSuccessHandler(logoutSuccessHandler);
    }

    private void authorizeAnyRequest(
            AuthorizeHttpRequestsConfigurer<HttpSecurity>.AuthorizationManagerRequestMatcherRegistry urlRegistry) {
        urlRegistry.anyRequest().permitAll();
    }

    private OidcUser keycloakUserMapper(OidcUserRequest userRequest,
            OidcUserInfo userInfo) {
        var authorities = mapAuthorities(userRequest, userInfo);
        var providerDetails = userRequest.getClientRegistration()
                .getProviderDetails();
        var userNameAttributeName = providerDetails.getUserInfoEndpoint()
                .getUserNameAttributeName();
        if (StringUtils.hasText(userNameAttributeName)) {
            return new DefaultOidcUser(authorities, userRequest.getIdToken(),
                    userInfo, userNameAttributeName);
        }
        return new DefaultOidcUser(authorities, userRequest.getIdToken(),
                userInfo);
    }

    /**
     * Maps authorities from {@link OidcUserRequest} and {@link OidcUserInfo} to
     * a collection of {@link GrantedAuthority} that includes Keycloak's realm
     * and client roles.
     *
     * @param userRequest
     *            the request containing the ID token and access token
     * @param userInfo
     *            the user info containing the user attributes
     * @return a collection of {@link GrantedAuthority}
     */
    public static Collection<? extends GrantedAuthority> mapAuthorities(
            OidcUserRequest userRequest, OidcUserInfo userInfo) {
        var authorities = new LinkedHashSet<GrantedAuthority>();
        authorities
                .add(new OidcUserAuthority(userRequest.getIdToken(), userInfo));

        var accessToken = userRequest.getAccessToken();
        for (var scope : accessToken.getScopes()) {
            authorities.add(new SimpleGrantedAuthority(SCOPE_PREFIX + scope));
        }

        var clientRegistration = userRequest.getClientRegistration();
        var issuer = clientRegistration.getProviderDetails().getIssuerUri();
        var jwtDecoder = NimbusJwtDecoder
                .withJwkSetUri(
                        clientRegistration.getProviderDetails().getJwkSetUri())
                .build();
        jwtDecoder
                .setJwtValidator(JwtValidators.createDefaultWithIssuer(issuer));
        var jwt = jwtDecoder.decode(accessToken.getTokenValue());

        if (jwt.hasClaim(REALM_ACCESS_CLAIM)) {
            var realmAccessClaim = MAPPER.convertValue(
                    jwt.getClaimAsMap(REALM_ACCESS_CLAIM),
                    new TypeReference<Map<String, List<String>>>() {
                    });
            var roles = realmAccessClaim.getOrDefault(ROLES_CLAIM, List.of());
            roles.stream().map(String.class::cast).map(KeycloakRealmRole::new)
                    .forEach(authorities::add);
        }

        if (jwt.hasClaim(RESOURCE_ACCESS_CLAIM)) {
            var resourceAccessClaim = MAPPER.convertValue(
                    jwt.getClaimAsMap(RESOURCE_ACCESS_CLAIM),
                    new TypeReference<Map<String, Map<String, List<String>>>>() {
                    });
            var clientId = clientRegistration.getClientId();
            var clientResources = resourceAccessClaim.getOrDefault(clientId,
                    Map.of());
            var roles = clientResources.getOrDefault(ROLES_CLAIM, List.of());
            roles.stream().map(String.class::cast)
                    .map(role -> new KeycloakClientRole(clientId, role))
                    .forEach(authorities::add);
        }
        return Set.copyOf(authorities);
    }

    static class KeycloakRealmRole implements GrantedAuthority {

        private final String role;

        public KeycloakRealmRole(String role) {
            Assert.hasText(role, "Role must not be empty");
            this.role = role;
        }

        public String getRole() {
            return role;
        }

        @Override
        public String getAuthority() {
            return ROLE_PREFIX + role;
        }

        @Override
        public int hashCode() {
            return Objects.hash(role);
        }

        @Override
        public boolean equals(Object other) {
            if (other instanceof KeycloakRealmRole krr) {
                return Objects.equals(role, krr.role);
            }
            return false;
        }
    }

    static class KeycloakClientRole implements GrantedAuthority {

        private final String client;

        private final String role;

        public KeycloakClientRole(String client, String role) {
            Assert.hasText(client, "Client must not be empty");
            Assert.hasText(role, "Role must not be empty");
            this.client = client;
            this.role = role;
        }

        public String getClient() {
            return client;
        }

        public String getRole() {
            return role;
        }

        @Override
        public String getAuthority() {
            return ROLE_PREFIX + role;
        }

        @Override
        public int hashCode() {
            return Objects.hash(client, role);
        }

        @Override
        public boolean equals(Object obj) {
            if (obj instanceof KeycloakClientRole kcr) {
                return Objects.equals(client, kcr.client)
                        && Objects.equals(role, kcr.role);
            }
            return false;
        }
    }
}
