/*-
 * Copyright (C) 2024 Vaadin Ltd
 *
 * This program is available under Vaadin Commercial License and Service Terms.
 *
 * See <https://vaadin.com/commercial-license-and-service-terms> for the full
 *  license.
-*/
package com.vaadin.controlcenter.starter.idm;

import java.util.HashMap;
import java.util.Map;

import org.springframework.boot.autoconfigure.AutoConfiguration;
import org.springframework.boot.autoconfigure.condition.ConditionalOnCloudPlatform;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.boot.cloud.CloudPlatform;
import org.springframework.boot.web.client.RestTemplateBuilder;
import org.springframework.cloud.context.config.annotation.RefreshScope;
import org.springframework.context.annotation.Bean;
import org.springframework.core.ParameterizedTypeReference;
import org.springframework.http.RequestEntity;
import org.springframework.security.oauth2.client.registration.ClientRegistration;
import org.springframework.security.oauth2.client.registration.ClientRegistrationRepository;
import org.springframework.security.oauth2.client.registration.ClientRegistrations;
import org.springframework.security.oauth2.client.registration.InMemoryClientRegistrationRepository;
import org.springframework.web.client.RestTemplate;

@AutoConfiguration
class ClientRegistrationRepositoryConfiguration {

    private static final ParameterizedTypeReference<Map<String, Object>> MAP_TYPE_REFERENCE = new ParameterizedTypeReference<>() {
    };

    private final RestTemplate rest;

    ClientRegistrationRepositoryConfiguration(RestTemplateBuilder builder) {
        this.rest = builder.build();
    }

    /*
     * This method overrides the default bean provider in {@link
     * OAuth2ClientRegistrationRepositoryConfiguration#
     * clientRegistrationRepository(OAuth2ClientProperties)} that is conditional
     * to already configured client registrations.
     *
     * Conditions are not re-evaluated at refresh time, so we need to provide a
     * new bean that is not conditional to the presence of client registrations.
     */
    @Bean
    @RefreshScope
    @ConditionalOnMissingBean
    @ConditionalOnCloudPlatform(CloudPlatform.KUBERNETES)
    ClientRegistrationRepository clientRegistrationRepository(
            IdentityManagementProperties properties) {
        var registrations = new HashMap<String, ClientRegistration>();
        var issuerBackendUri = properties.getIssuerBackendUri();
        if (issuerBackendUri != null) {
            var issuerFrontendUri = properties.getIssuerFrontendUri();
            var clientId = properties.getClientId();
            var clientSecret = properties.getClientSecret();
            var clientRegistrationId = properties.getClientRegistrationId();
            var configuration = getConfiguration(issuerBackendUri);
            if (configuration == null) {
                var message = "Failed to retrieve configuration from %s";
                throw new IllegalStateException(
                        message.formatted(issuerBackendUri));
            }
            var metadataIssuer = configuration.get("issuer");
            if (!issuerFrontendUri.equals(metadataIssuer)) {
                var message = "The Issuer %s provided by the configuration "
                        + "did not match the expected issuer %s";
                throw new IllegalStateException(
                        message.formatted(metadataIssuer, issuerFrontendUri));
            }
            var registration = ClientRegistrations
                    .fromOidcConfiguration(configuration)
                    .registrationId(clientRegistrationId).clientId(clientId)
                    .clientSecret(clientSecret).scope("openid").build();
            registrations.put(clientRegistrationId, registration);
        }
        var delegate = new InMemoryClientRegistrationRepository(registrations);
        return new DelegatingClientRegistrationRepository(delegate);
    }

    private Map<String, Object> getConfiguration(String issuerUri) {
        var metadataEndpoint = issuerUri + "/.well-known/openid-configuration";
        var request = RequestEntity.get(metadataEndpoint).build();
        return rest.exchange(request, MAP_TYPE_REFERENCE).getBody();
    }
}
