/*-
 * Copyright (C) 2024 Vaadin Ltd
 *
 * This program is available under Vaadin Commercial License and Service Terms.
 *
 * See <https://vaadin.com/commercial-license-and-service-terms> for the full license.
-*/
package com.vaadin.controlcenter.starter.i18n;

import java.text.MessageFormat;
import java.util.Arrays;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.regex.Pattern;

import io.fabric8.kubernetes.api.model.ConfigMap;
import io.fabric8.kubernetes.client.KubernetesClient;
import io.fabric8.kubernetes.client.Watcher;
import io.fabric8.kubernetes.client.WatcherException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.boot.context.event.ApplicationReadyEvent;
import org.springframework.context.ApplicationListener;
import org.springframework.core.env.Environment;

import com.vaadin.flow.i18n.I18NProvider;
import com.vaadin.flow.server.VaadinRequest;

/**
 * An I18N provider that is used to preview translations from Control Center in
 * a Vaadin app. It loads translations from Kubernetes ConfigMaps, which are
 * provided by the Control Center app.
 */
public class ControlCenterI18NProvider
        implements I18NProvider, ApplicationListener<ApplicationReadyEvent> {
    static final String PREVIEW_APPLICATION_LABEL = "vaadin.com/owned-by";
    static final String PREVIEW_MARKER_LABEL = "vaadin.com/i18n.preview";
    static final String PREVIEW_LANGUAGE_LABEL = "vaadin.com/i18n.preview.language";
    static final String PREVIEW_DEFAULT_LANGUAGE_LABEL = "vaadin.com/i18n.preview.default-language";
    static final String I18N_PREVIEW_COOKIE = "i18n-preview";

    private static final Pattern INVALID_CONFIG_KEY_CHARS_PATTERN = Pattern
            .compile("[^-._a-zA-Z0-9]");

    private final Map<Locale, Translation> translations = new HashMap<>();
    private Locale defaultLanguage;
    private final I18NProvider defaultProvider;

    private final transient KubernetesClient client;

    private transient Environment environment;

    ControlCenterI18NProvider(KubernetesClient client,
            I18NProvider defaultProvider) {
        this.client = client;
        this.defaultProvider = defaultProvider;
    }

    @Override
    public void onApplicationEvent(ApplicationReadyEvent event) {
        environment = event.getApplicationContext().getEnvironment();
        watchConfigMaps(getConfigMapWatcher());
    }

    void watchConfigMaps(Watcher<ConfigMap> watcher) {
        var applicationId = environment.getProperty("spring.application.name");
        if (applicationId == null || applicationId.isEmpty()) {
            getLogger().warn(
                    "No application ID set. Not watching ConfigMaps for translations.");
            return;
        }

        try {
            getLogger().info("Watching ConfigMaps for application {}",
                    applicationId);
            client.configMaps().inNamespace(client.getNamespace())
                    .withLabel(PREVIEW_MARKER_LABEL)
                    .withLabel(PREVIEW_APPLICATION_LABEL, applicationId)
                    .watch(watcher);
        } catch (Exception e) {
            getLogger().error("Error watching ConfigMaps", e);
        }
    }

    private Watcher<ConfigMap> getConfigMapWatcher() {
        return new Watcher<>() {
            @Override
            public void eventReceived(Action action, ConfigMap resource) {
                switch (action) {
                case ADDED, MODIFIED -> addOrUpdateTranslation(resource);
                case DELETED -> removeTranslation(resource);
                default -> {
                    // Ignore, no need to handle other actions
                }
                }
            }

            @Override
            public void onClose(WatcherException cause) {
                getLogger().error(
                        "ConfigMaps watcher closed with: {}. Retrying...",
                        cause.getMessage(), cause);
                watchConfigMaps(getConfigMapWatcher());
            }
        };
    }

    void addOrUpdateTranslation(ConfigMap resource) {
        getLogger().info("Adding or updating translation for ConfigMap {}",
                resource.getMetadata().getName());
        var locale = detectLocale(resource);

        if (locale == null) {
            return;
        }

        var isDefault = isDefaultLanguage(resource);

        if (isDefault) {
            defaultLanguage = locale;
        }

        translations.put(locale,
                new Translation(locale, new HashMap<>(resource.getData())));
    }

    private void removeTranslation(ConfigMap resource) {
        var locale = detectLocale(resource);
        translations.remove(locale);

        if (Objects.equals(locale, defaultLanguage)) {
            defaultLanguage = null;
        }
    }

    private boolean isDefaultLanguage(ConfigMap map) {
        return String.valueOf(true).equals(map.getMetadata().getLabels()
                .get(PREVIEW_DEFAULT_LANGUAGE_LABEL));
    }

    private Locale detectLocale(ConfigMap configMap) {
        var languageTag = configMap.getMetadata().getLabels()
                .get(PREVIEW_LANGUAGE_LABEL);

        if (languageTag == null || languageTag.isEmpty()) {
            getLogger().warn(
                    "ConfigMap {} does not have a language tag label. Ignoring.",
                    configMap.getMetadata().getName());
            return null;
        }

        // Make sure the language tag is in the correct format
        var sanitizedLanguageTag = languageTag.replace("_", "-");

        return Locale.forLanguageTag(sanitizedLanguageTag);
    }

    private Optional<Translation> resolveLanguage(Locale locale) {
        // Look for specified locale first
        var language = Optional.ofNullable(translations.get(locale));

        // Use a locale with the same language as a fallback
        language = language.or(() -> translations.keySet().stream()
                .filter(l -> l.getLanguage().equals(locale.getLanguage()))
                .min(Comparator.comparing(Locale::toLanguageTag))
                .map(translations::get));

        // Use the default language as a last resort
        return language.or(
                () -> Optional.ofNullable(translations.get(defaultLanguage)));
    }

    /**
     * Config maps only allow certain characters in keys, whereas a message key
     * coming from a properties file may contain other characters. There's also
     * a max length of 253 characters. This is a best effort to generate a valid
     * config map key from a message key.
     *
     * @param messageKey
     *            the message key
     * @return the sanitized message key
     */
    private String generateConfigMapKey(String messageKey) {
        messageKey = INVALID_CONFIG_KEY_CHARS_PATTERN.matcher(messageKey)
                .replaceAll("_");
        return messageKey.substring(0, Math.min(messageKey.length(), 253));
    }

    @Override
    public List<Locale> getProvidedLocales() {
        // Try to return the locales in a predictable order
        // Default language first, then the rest sorted by language tag
        var providedLocales = new HashSet<Locale>();

        if (isPreviewEnabled()) {
            providedLocales.addAll(translations.keySet());
        } else if (defaultProvider != null) {
            return defaultProvider.getProvidedLocales();
        }

        return providedLocales.stream().sorted((l1, l2) -> {
            if (l1.equals(defaultLanguage)) {
                return -1;
            }
            if (l2.equals(defaultLanguage)) {
                return 1;
            }
            return l1.toLanguageTag().compareTo(l2.toLanguageTag());
        }).toList();
    }

    @Override
    public String getTranslation(String key, Locale locale, Object... params) {
        if (isPreviewEnabled()) {
            var maybePreviewLanguage = resolveLanguage(locale);

            if (maybePreviewLanguage.isEmpty()) {
                return getTranslationNotFound(key, locale);
            }

            var configMapKey = generateConfigMapKey(key);
            var previewTranslation = maybePreviewLanguage.get();
            var translation = previewTranslation.translations()
                    .get(configMapKey);

            if (translation == null) {
                return getTranslationNotFound(key, locale);
            }

            if (params.length > 0) {
                translation = new MessageFormat(translation,
                        previewTranslation.locale()).format(params);
            }

            return translation;
        } else {
            if (defaultProvider == null) {
                return getTranslationNotFound(key, locale);
            }
            return defaultProvider.getTranslation(key, locale, params);
        }
    }

    private boolean isPreviewEnabled() {
        VaadinRequest request = VaadinRequest.getCurrent();
        if (request == null || request.getCookies() == null) {
            return false;
        }
        return Arrays.stream(request.getCookies())
                .filter(cookie -> I18N_PREVIEW_COOKIE.equals(cookie.getName()))
                .findFirst()
                .map(cookie -> Boolean.parseBoolean(cookie.getValue()))
                .orElse(false);
    }

    private static String getTranslationNotFound(String key, Locale locale) {
        getLogger().debug("Missing translation for locale {}", locale);
        return "!" + locale.toLanguageTag() + ": " + key;
    }

    static Logger getLogger() {
        return LoggerFactory.getLogger(ControlCenterI18NProvider.class);
    }

    record Translation(Locale locale, Map<String, String> translations) {
    }
}
