/*-
 * Copyright (C) 2024 Vaadin Ltd
 *
 * This program is available under Vaadin Commercial License and Service Terms.
 *
 * See <https://vaadin.com/commercial-license-and-service-terms> for the full license.
-*/
package com.vaadin.controlcenter.starter.i18n;

import io.fabric8.kubernetes.client.KubernetesClient;
import org.jetbrains.annotations.NotNull;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.config.BeanPostProcessor;
import org.springframework.boot.autoconfigure.AutoConfiguration;
import org.springframework.boot.autoconfigure.AutoConfigureBefore;
import org.springframework.boot.autoconfigure.condition.ConditionalOnCloudPlatform;
import org.springframework.boot.cloud.CloudPlatform;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.boot.web.servlet.FilterRegistrationBean;
import org.springframework.context.annotation.Bean;
import org.springframework.context.support.AbstractApplicationContext;

import com.vaadin.flow.i18n.I18NProvider;
import com.vaadin.flow.spring.SpringBootAutoConfiguration;

/**
 * This configuration bean configures the Control Center I18N provider when the
 * respective profile is enabled. It is set up to configure before the Vaadin
 * Spring Boot autoconfiguration in order to ensure the custom I18N provider
 * bean is registered before the Vaadin config checks for existing I18N provider
 * beans.
 */
@AutoConfiguration
@AutoConfigureBefore(SpringBootAutoConfiguration.class)
@EnableConfigurationProperties(ControlCenterI18NProperties.class)
public class ControlCenterI18NConfiguration {

    /**
     * Creates a new BeanPostProcessor bean, which replaces the current
     * I18NProvider with a ControlCenterI18NProvider.
     *
     * @param client
     *            the Kubernetes client
     * @param applicationContext
     *            the application context to which the ControlCenterI18NProvider
     *            will be added as an application listener
     * @return the BeanPostProcessor bean
     */
    @Bean
    @ConditionalOnCloudPlatform(CloudPlatform.KUBERNETES)
    static BeanPostProcessor i18nProviderBeanPostProcessor(
            KubernetesClient client,
            AbstractApplicationContext applicationContext) {
        return new BeanPostProcessor() {
            @Override
            public Object postProcessAfterInitialization(@NotNull Object bean,
                    @NotNull String beanName) throws BeansException {
                if (bean instanceof I18NProvider i18NProvider) {
                    var ccI18NProvider = new ControlCenterI18NProvider(client,
                            i18NProvider);
                    applicationContext.addApplicationListener(ccI18NProvider);
                    return ccI18NProvider;
                }
                return bean;
            }
        };
    }

    /**
     * Creates a filter registration bean that adds the
     * {@code I18NPreviewFilter}.
     *
     * @return the filter registration bean
     */
    @Bean
    @ConditionalOnCloudPlatform(CloudPlatform.KUBERNETES)
    FilterRegistrationBean<I18NPreviewFilter> i18nPreviewFilter() {
        FilterRegistrationBean<I18NPreviewFilter> registration = new FilterRegistrationBean<>();
        registration.setFilter(new I18NPreviewFilter());
        registration.addUrlPatterns("/*");
        return registration;
    }

    @Bean
    @ConditionalOnCloudPlatform(CloudPlatform.KUBERNETES)
    I18NPreviewServiceInitListener i18NPreviewServiceInitListener() {
        return new I18NPreviewServiceInitListener();
    }
}
