// Copyright (C) 2025 Vaadin Ltd
// This program is available under Vaadin Commercial License and Service Terms.
// See <https://vaadin.com/commercial-license-and-service-terms> for the full license.
package com.vaadin.controlcenter.aot;

import java.util.Set;

import io.fabric8.kubernetes.client.KubernetesClient;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.aot.BeanRegistrationAotContribution;
import org.springframework.beans.factory.aot.BeanRegistrationAotProcessor;
import org.springframework.beans.factory.aot.BeanRegistrationExcludeFilter;
import org.springframework.beans.factory.support.RegisteredBean;
import org.springframework.cloud.kubernetes.commons.KubernetesClientProperties;
import org.springframework.lang.Nullable;

/// This [BeanRegistrationExcludeFilter] excludes beans from AOT processing that are registered programmatically during
/// runtime, such as [KubernetesClient] and [KubernetesClientProperties]. This is necessary since the AOT processor is
/// not aware of programmatic bean registration at compile time and would otherwise generate registration methods for
/// these beans, which would lead to errors during context initialization.
class SpringCloudBeanRegistrationProcessor implements BeanRegistrationExcludeFilter, BeanRegistrationAotProcessor {

    private static final Logger LOGGER = LoggerFactory.getLogger(SpringCloudBeanRegistrationProcessor.class);

    SpringCloudBeanRegistrationProcessor() {}

    @Override
    public boolean isExcludedFromAotProcessing(RegisteredBean registeredBean) {
        LOGGER.trace("Checking if bean is excluded from AOT processing: {}", registeredBean.getBeanName());
        return Set.of(KubernetesClient.class, KubernetesClientProperties.class).contains(registeredBean.getBeanClass());
    }

    @Override
    public BeanRegistrationAotContribution processAheadOfTime(@Nullable RegisteredBean registeredBean) {
        return (generationContext, beanRegistrationCode) -> {};
    }
}
