// Copyright (C) 2025 Vaadin Ltd
// This program is available under Vaadin Commercial License and Service Terms.
// See <https://vaadin.com/commercial-license-and-service-terms> for the full license.
package com.vaadin.controlcenter.aot;

import java.util.Map;

import org.springframework.boot.SpringApplication;
import org.springframework.boot.env.EnvironmentPostProcessor;
import org.springframework.context.aot.AbstractAotProcessor;
import org.springframework.core.env.ConfigurableEnvironment;
import org.springframework.core.env.MapPropertySource;

/// Environment post processor that enables Spring Cloud Kubernetes configuration.
class SpringCloudEnvironmentPostProcessor implements EnvironmentPostProcessor {

    private static final Map<String, Object> PROPERTIES = Map.of(
            // Include configurations conditional on the service-name property in ahead-of-time processing
            "vaadin.control-center.service-name", "hazelcast",
            // Include configurations conditional on CloudPlatform.KUBERNETES in ahead-of-time processing
            "spring.main.cloud-platform", "kubernetes",
            // Expose actuator endpoints to register the beans during AOT
            "management.endpoints.web.exposure.include", "beans,flyway,health,info,metrics,vaadin",
            // Set the management server port to 8081 during AOT (can be overridden at runtime)
            "management.server.port", "8081");

    @Override
    public void postProcessEnvironment(ConfigurableEnvironment environment, SpringApplication application) {
        if (Boolean.getBoolean(AbstractAotProcessor.AOT_PROCESSING)) {
            environment.setActiveProfiles("kubernetes");
            environment.getPropertySources().addFirst(new MapPropertySource("enableKubernetes", PROPERTIES));
        }
    }
}
