// Copyright (C) 2025 Vaadin Ltd
// This program is available under Vaadin Commercial License and Service Terms.
// See <https://vaadin.com/commercial-license-and-service-terms> for the full license.
package com.vaadin.controlcenter.aot;

import io.github.classgraph.ClassGraph;
import org.jboss.resteasy.client.jaxrs.internal.ResteasyClientBuilderImpl;
import org.jboss.resteasy.client.jaxrs.internal.proxy.ProxyBuilderImpl;
import org.jboss.resteasy.client.jaxrs.internal.proxy.ResteasyClientProxy;
import org.keycloak.admin.client.JacksonProvider;
import org.keycloak.admin.client.spi.ResteasyClientClassicProvider;
import org.springframework.aot.hint.BindingReflectionHintsRegistrar;
import org.springframework.aot.hint.MemberCategory;
import org.springframework.aot.hint.ProxyHints;
import org.springframework.aot.hint.ReflectionHints;
import org.springframework.aot.hint.RuntimeHints;
import org.springframework.aot.hint.RuntimeHintsRegistrar;
import org.springframework.aot.hint.TypeHint;
import org.springframework.aot.hint.TypeReference;

class KeycloakRuntimeHints implements RuntimeHintsRegistrar {

    private static final ClassGraph GRAPH = new ClassGraph();

    private static final BindingReflectionHintsRegistrar BINDING_REGISTRAR = new BindingReflectionHintsRegistrar();

    private static final String[] BINDING_PACKAGES = {
        "org.keycloak.representations.idm", "org.keycloak.representations.userprofile.config"
    };

    private static final String[] PROXY_PACKAGES = {
        "org.keycloak.admin.client.token", "org.keycloak.admin.client.resource"
    };

    private static final Class<?>[] REFLECTION_CLASSES = {
        org.jboss.resteasy.plugins.providers.multipart.i18n.LogMessages_$logger.class,
        org.jboss.resteasy.plugins.providers.jaxb.i18n.LogMessages_$logger.class,
        org.jboss.resteasy.resteasy_jaxrs.i18n.LogMessages_$logger.class,
        org.jboss.resteasy.client.jaxrs.i18n.LogMessages_$logger.class,
        org.jboss.resteasy.resteasy_jaxrs.i18n.Messages_$bundle.class,
        org.jboss.resteasy.client.jaxrs.i18n.Messages_$bundle.class,
        ResteasyClientClassicProvider.class,
        ResteasyClientBuilderImpl.class,
        ProxyBuilderImpl.class,
        JacksonProvider.class
    };

    private static final MemberCategory[] REFLECTION_HINTS = {
        MemberCategory.INVOKE_PUBLIC_CONSTRUCTORS, MemberCategory.INVOKE_PUBLIC_METHODS, MemberCategory.DECLARED_FIELDS
    };

    KeycloakRuntimeHints() {}

    @Override
    public void registerHints(RuntimeHints hints, ClassLoader classLoader) {
        registerReflectionHints(hints.reflection());
        registerProxyHints(hints.proxies());
    }

    private void registerReflectionHints(ReflectionHints hints) {
        var representations = GRAPH.acceptPackages(BINDING_PACKAGES);
        try (var scan = representations.scan()) {
            var types = scan.getAllStandardClasses().loadClasses().toArray(Class[]::new);
            BINDING_REGISTRAR.registerReflectionHints(hints, types);
        }
        hints.registerTypes(TypeReference.listOf(REFLECTION_CLASSES), TypeHint.builtWith(REFLECTION_HINTS));
    }

    private void registerProxyHints(ProxyHints hints) {
        var resources = GRAPH.acceptPackages(PROXY_PACKAGES);
        try (var scan = resources.scan()) {
            scan.getAllInterfaces()
                    .loadClasses()
                    .forEach(resource -> hints.registerJdkProxy(resource, ResteasyClientProxy.class));
        }
    }
}
