/*
 * Copyright 2000-2026 Vaadin Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
package com.vaadin.flow.component.confirmdialog;

import java.util.NoSuchElementException;

import com.vaadin.browserless.ComponentTester;
import com.vaadin.browserless.Tests;
import com.vaadin.flow.component.ComponentUtil;
import com.vaadin.flow.dom.Element;

/**
 * Tester for ConfirmDialog.
 */
@Tests(ConfirmDialog.class)
public class ConfirmDialogTester extends ComponentTester<ConfirmDialog> {
    /**
     * Wrap given component for testing.
     *
     * @param component
     *            target component
     */
    public ConfirmDialogTester(ConfirmDialog component) {
        super(component);
    }

    public void open() {
        getComponent().open();
        roundTrip();
    }

    /**
     * Click the confirm button.
     */
    public void confirm() {
        ensureComponentIsUsable();

        ComponentUtil.fireEvent(getComponent(),
                new ConfirmDialog.ConfirmEvent(getComponent(), true));
        getComponent().close();
    }

    /**
     * Click cancel button.
     *
     * @throws IllegalStateException
     *             when cancel button is not enabled
     */
    public void cancel() {
        ensureComponentIsUsable();
        if (!Boolean.parseBoolean(getComponent().getElement()
                .getProperty("cancelButtonVisible"))) {
            throw new IllegalStateException("Cancel button is not available.");
        }
        ComponentUtil.fireEvent(getComponent(),
                new ConfirmDialog.CancelEvent(getComponent(), true));
        getComponent().close();
    }

    /**
     * Click reject button.
     *
     * @throws IllegalStateException
     *             when reject button is not enabled
     */
    public void reject() {
        ensureComponentIsUsable();
        if (!Boolean.parseBoolean(getComponent().getElement()
                .getProperty("rejectButtonVisible"))) {
            throw new IllegalStateException("Reject button is not available.");
        }
        ComponentUtil.fireEvent(getComponent(),
                new ConfirmDialog.RejectEvent(getComponent(), true));
        getComponent().close();
    }

    /**
     * Get the confirmation message text set to the component.
     *
     * @return confirmation message
     */
    public String getText() {
        return getComponent().getElement().getProperty("message");
    }

    /**
     * Get the header of the confirm dialog.
     *
     * @return current header of dialog
     */
    public String getHeader() {
        return getComponent().getElement().getProperty("header");
    }

    /**
     * Get the header element set to the confirm dialog.
     *
     * @return header element
     * @throws NoSuchElementException
     *             if no header element found
     */
    public Element getHeaderElement() {
        return getComponent().getElement().getChildren()
                .filter(elem -> "header".equals(elem.getAttribute("slot")))
                .findFirst().orElseThrow(() -> new IllegalStateException(
                        "No header element set"));
    }
}
