/*-
 * Copyright (C) 2023 Vaadin Ltd
 *
 * This program is available under Vaadin Commercial License and Service Terms.
 *
 *
 * See <https://vaadin.com/commercial-license-and-service-terms> for the full license.
 */
package com.vaadin.appsec.views;

import java.text.DateFormat;
import java.time.Instant;
import java.util.Date;
import java.util.List;
import java.util.Objects;

import org.jsoup.Jsoup;
import org.jsoup.safety.Safelist;

import com.vaadin.appsec.backend.AppSecService;
import com.vaadin.appsec.backend.model.AppSecData;
import com.vaadin.appsec.backend.model.analysis.AffectedVersion;
import com.vaadin.appsec.backend.model.analysis.AssessmentStatus;
import com.vaadin.appsec.backend.model.dto.SeverityLevel;
import com.vaadin.appsec.backend.model.dto.Vulnerability;
import com.vaadin.flow.component.ClickEvent;
import com.vaadin.flow.component.Component;
import com.vaadin.flow.component.HasValue;
import com.vaadin.flow.component.Html;
import com.vaadin.flow.component.UI;
import com.vaadin.flow.component.Unit;
import com.vaadin.flow.component.button.Button;
import com.vaadin.flow.component.button.ButtonVariant;
import com.vaadin.flow.component.combobox.ComboBox;
import com.vaadin.flow.component.html.NativeLabel;
import com.vaadin.flow.component.html.Span;
import com.vaadin.flow.component.icon.Icon;
import com.vaadin.flow.component.icon.VaadinIcon;
import com.vaadin.flow.component.orderedlayout.HorizontalLayout;
import com.vaadin.flow.component.orderedlayout.VerticalLayout;
import com.vaadin.flow.component.textfield.TextArea;
import com.vaadin.flow.server.Version;
import com.vaadin.flow.theme.lumo.LumoUtility;

/**
 * Vulnerability details view contains details of a vulnerability.
 */
public class VulnerabilityDetailsView extends VerticalLayout {

    private final DateFormat dateFormatter = DateFormat
            .getDateInstance(DateFormat.DEFAULT, UI.getCurrent().getLocale());
    private final DateFormat dateTimeFormatter = DateFormat.getDateTimeInstance(
            DateFormat.DEFAULT, DateFormat.DEFAULT,
            UI.getCurrent().getLocale());
    private final Vulnerability vulnerabilityDTO;
    private final Runnable closeCallback;
    private Button saveButton;
    private ComboBox<AppSecData.VulnerabilityStatus> developerStatus;
    private TextArea developerAnalysis;
    private Span lastUpdated;

    public VulnerabilityDetailsView(Vulnerability vulnerabilityDTO,
            Runnable closeCallback) {
        this.vulnerabilityDTO = vulnerabilityDTO;
        this.closeCallback = closeCallback;
        setSizeFull();
        setMargin(false);
        addClassName("vulnerability-details-view");
        buildHeaderBar();
        buildContent();
    }

    private void buildHeaderBar() {
        HorizontalLayout header = new HorizontalLayout();
        header.addClassName("vulnerability-header");
        header.setDefaultVerticalComponentAlignment(Alignment.BASELINE);
        header.setWidth(100, Unit.PERCENTAGE);

        Button backButton = new Button(new Icon(VaadinIcon.CHEVRON_LEFT));
        backButton.addClassName("navigate-back-button");
        backButton.addThemeVariants(ButtonVariant.LUMO_ICON);
        backButton.addClassName(LumoUtility.Border.NONE);
        backButton.addClassName(LumoUtility.Background.TRANSPARENT);
        backButton.setAriaLabel("Navigate back");
        backButton.setTooltipText("Navigate back");
        backButton.addClickListener(e -> closeCallback.run());
        header.add(backButton);

        Span vulnerabilityTitle = new Span(vulnerabilityDTO.getIdentifier());
        vulnerabilityTitle.addClassName("vulnerability-title");
        header.addAndExpand(vulnerabilityTitle);

        add(header);
    }

    private void buildContent() {
        Component leftContent = buildLeftContent();
        Component rightContent = buildRightContent();
        HorizontalLayout content = new HorizontalLayout(leftContent,
                rightContent);
        content.expand(leftContent);
        content.setSizeFull();
        content.addClassName("vulnerability-details-content-panel");
        addAndExpand(content);
    }

    private Component buildLeftContent() {
        VerticalLayout left = new VerticalLayout();
        left.addClassName("vulnerability-details-content-left");
        left.setMargin(true);

        left.add(buildProperties());
        left.add(buildVaadinAnalysis());
        left.add(buildVulnerabilityDesc());
        left.add(buildReferences());

        HorizontalLayout leftPanel = new HorizontalLayout(left);
        leftPanel.setSizeFull();
        leftPanel.addClassName(LumoUtility.Border.NONE);

        return leftPanel;
    }

    private Component buildProperties() {
        HorizontalLayout properties = new HorizontalLayout();
        properties.setWidth(100, Unit.PERCENTAGE);

        properties.add(buildLabel("Ecosystem",
                vulnerabilityDTO.getDependency().getEcosystem().toString()));
        properties.add(buildLabel("Dependency",
                vulnerabilityDTO.getDependency().toString()));
        properties.add(buildLabel("Version",
                vulnerabilityDTO.getDependency().getVersion()));
        properties.add(buildLabel("Patched in version",
                vulnerabilityDTO.getPatchedVersion()));
        properties.add(buildLabel("Risk rating",
                String.valueOf(vulnerabilityDTO.getRiskScore())));
        properties.add(buildLabel("Status", "---"));
        properties.add(buildLabel("Time of detection",
                dateFormatter.format(vulnerabilityDTO.getDatePublished())));
        SeverityLevel severityLevel = vulnerabilityDTO.getSeverityLevel();
        Component severity = buildLabel("Severity", severityLevel
                + (SeverityLevel.NONE == severityLevel ? "" : " severity"));
        severity.addClassName(
                "severity-" + vulnerabilityDTO.getSeverityLevel().name());
        properties.add(severity);

        return properties;
    }

    private Component buildLabel(String caption, String value) {
        Span propertyValue = new Span(value);
        propertyValue.setId(caption);

        NativeLabel propertyLabel = new NativeLabel(caption);
        propertyLabel.setFor(propertyValue);
        propertyLabel.addClassName("vulnerability-property-label");

        VerticalLayout layout = new VerticalLayout(propertyLabel,
                propertyValue);
        layout.addClassName("vulnerability-details-property-holder");
        layout.setSizeUndefined();

        return layout;
    }

    private Component buildVaadinAnalysis() {
        VerticalLayout vaadinAnalysisPanel = new VerticalLayout();
        vaadinAnalysisPanel.addClassName("vaadin-analysis-panel");
        vaadinAnalysisPanel.addClassName(LumoUtility.Margin.SMALL);

        Span vaadinAnalysisTitle = new Span("Vaadin analysis");
        vaadinAnalysisTitle.addClassName("vaadin-analysis-title");
        vaadinAnalysisTitle.setWidth(100, Unit.PERCENTAGE);
        vaadinAnalysisPanel.add(vaadinAnalysisTitle);

        String flowVersion = Version.getFullVersion();
        AppSecService service = AppSecService.getInstance();
        List<String> flowVersions = service.getSupportedFlowVersions();
        boolean isSupportedVaadinVersion = flowVersions.contains(flowVersion);

        Span vaadinAnalysisStatus = new Span();
        Span vaadinAnalysisDesc = new Span();
        if (!isSupportedVaadinVersion) {
            vaadinAnalysisStatus.setText("Not Provided");
            vaadinAnalysisStatus.addClassName("vaadin-analysis-not-supported");
            vaadinAnalysisDesc.setText("This app is running with Vaadin Flow "
                    + flowVersion
                    + " which is not currently getting security updates. Please "
                    + " upgrade to one of the latest maintained versions: "
                    + String.join(", ", flowVersions));
        } else {
            AffectedVersion affectedVersion = vulnerabilityDTO
                    .getAffectedVersion();
            if (affectedVersion != null) {
                AssessmentStatus status = affectedVersion.getStatus();
                vaadinAnalysisStatus.setText(status.toString());
                setStatusStyle(vaadinAnalysisStatus, status);
                String comment = affectedVersion.getComment();
                vaadinAnalysisDesc.setText(comment);
            } else {
                vaadinAnalysisStatus.setText("Not Available");
                vaadinAnalysisDesc.setText(
                        "This vulnerability has not yet been assessed by "
                                + "the Vaadin Security Team. Please check later for updates.");
                vaadinAnalysisStatus
                        .addClassName("vaadin-analysis-not-available");
            }
        }
        vaadinAnalysisDesc.addClassName("vaadin-analysis-desc");
        vaadinAnalysisDesc.setWidth(100, Unit.PERCENTAGE);

        vaadinAnalysisPanel.add(vaadinAnalysisStatus);
        vaadinAnalysisPanel.add(vaadinAnalysisDesc);
        return vaadinAnalysisPanel;
    }

    private void setStatusStyle(Span vaadinAnalysisStatus,
            AssessmentStatus status) {
        switch (status) {
        case TRUE_POSITIVE -> vaadinAnalysisStatus
                .addClassName("vaadin-analysis-true-positive");
        case FALSE_POSITIVE -> vaadinAnalysisStatus
                .addClassName("vaadin-analysis-false-positive");
        case UNDER_REVIEW -> vaadinAnalysisStatus
                .addClassName("vaadin-analysis-under-review");
        default -> vaadinAnalysisStatus
                .addClassName("vaadin-analysis-not-supported");
        }
    }

    private Component buildVulnerabilityDesc() {
        VerticalLayout vulnerabilityDescPanel = new VerticalLayout();

        Span vulnerabilityDescTitle = new Span("Vulnerability description");
        vulnerabilityDescTitle.addClassName("vulnerability-desc-title");
        vulnerabilityDescTitle.setWidth(100, Unit.PERCENTAGE);
        vulnerabilityDescPanel.add(vulnerabilityDescTitle);

        String cleanedDetails = Jsoup.clean(vulnerabilityDTO.getDetails(),
                Safelist.relaxed());
        Html vulnerabilityDesc = new Html(
                ("<div>\n" + cleanedDetails + "\n</div>"));
        vulnerabilityDesc.addClassName("vulnerability-desc");
        vulnerabilityDescPanel.add(vulnerabilityDesc);

        return vulnerabilityDescPanel;
    }

    private Component buildReferences() {
        VerticalLayout referencesPanel = new VerticalLayout();

        Span referencesTitle = new Span("References");
        referencesTitle.addClassName("references-title");
        referencesTitle.setWidth(100, Unit.PERCENTAGE);
        referencesPanel.add(referencesTitle);

        List<Html> links = vulnerabilityDTO
                .getReferenceUrls().stream().map(url -> new Html("<a href=\""
                        + url + "\" target=\"_blank\">" + url + "</a>"))
                .toList();
        links.forEach(referencesPanel::add);

        return referencesPanel;
    }

    private Component buildRightContent() {
        VerticalLayout right = new VerticalLayout();
        right.addClassName("vulnerability-details-content-right");
        right.addClassName(LumoUtility.Margin.SMALL);
        right.setMargin(true);
        right.setHeight(100, Unit.PERCENTAGE);
        right.setWidth(400, Unit.PIXELS);

        right.add(buildDeveloperAnalysisTitle());
        right.add(buildDeveloperAnalysisDesc());
        right.add(buildDeveloperStatus());
        right.addAndExpand(buildDeveloperAnalysis());
        right.add(buildLastUpdated());
        right.add(buildSaveButton());
        right.setHorizontalComponentAlignment(Alignment.END, saveButton);

        return right;
    }

    private Component buildDeveloperAnalysisTitle() {
        Span developerAnalysisTitle = new Span("Developer analysis");
        developerAnalysisTitle.addClassName("developer-analysis-title");
        developerAnalysisTitle.setWidth(100, Unit.PERCENTAGE);
        return developerAnalysisTitle;
    }

    private Component buildDeveloperAnalysisDesc() {
        Span developerAnalysisDesc = new Span(
                "You can declare a status for this vulnerability if you have done some analysis on this vulnerability or want to perform an analysis on this vulnerability.");
        developerAnalysisDesc.setWidth(100, Unit.PERCENTAGE);
        developerAnalysisDesc.addClassName("developer-analysis-desc");
        return developerAnalysisDesc;
    }

    private Component buildDeveloperStatus() {
        developerStatus = new ComboBox<>();
        developerStatus.setItems(AppSecData.VulnerabilityStatus.NOT_SET,
                AppSecData.VulnerabilityStatus.NOT_AFFECTED,
                AppSecData.VulnerabilityStatus.FALSE_POSITIVE,
                AppSecData.VulnerabilityStatus.IN_TRIAGE,
                AppSecData.VulnerabilityStatus.EXPLOITABLE);
        developerStatus.setPlaceholder("Select status");
        developerStatus.setWidth(100, Unit.PERCENTAGE);
        developerStatus.setLabel("Vulnerability status");
        developerStatus.setValue(vulnerabilityDTO.getDeveloperStatus());
        developerStatus.addValueChangeListener(this::developerAnalysisChanged);
        return developerStatus;
    }

    private Component buildDeveloperAnalysis() {
        developerAnalysis = new TextArea("Description (optional)");
        developerAnalysis.setPlaceholder(
                "Add a description that describes the state of this vulnerability");
        developerAnalysis.setWidth(100, Unit.PERCENTAGE);
        if (vulnerabilityDTO.getDeveloperAnalysis() != null) {
            developerAnalysis.setValue(vulnerabilityDTO.getDeveloperAnalysis());
        }
        developerAnalysis
                .addValueChangeListener(this::developerAnalysisChanged);
        return developerAnalysis;
    }

    private void developerAnalysisChanged(HasValue.ValueChangeEvent<?> event) {
        saveButton.setEnabled(!Objects.equals(
                vulnerabilityDTO.getDeveloperStatus(),
                developerStatus.getValue())
                || !Objects.equals(vulnerabilityDTO.getDeveloperAnalysis(),
                        developerAnalysis.getValue()));
    }

    private Component buildLastUpdated() {
        String timestamp = vulnerabilityDTO.getDeveloperUpdated() == null ? "-"
                : dateTimeFormatter.format(
                        Date.from(vulnerabilityDTO.getDeveloperUpdated()));
        lastUpdated = new Span("Last updated: " + timestamp);
        lastUpdated.setWidth(310, Unit.PIXELS);
        return lastUpdated;
    }

    private Component buildSaveButton() {
        saveButton = new Button("Save");
        saveButton.setEnabled(false);
        saveButton.addThemeVariants(ButtonVariant.LUMO_PRIMARY);
        saveButton.setAriaLabel("Save developer analysis");
        saveButton.setTooltipText("Save developer analysis");
        saveButton.addClickListener(this::save);
        return saveButton;
    }

    private void save(ClickEvent<Button> clickEvent) {
        // Defaults to NOT_SET if not given by user
        if (developerStatus.getValue() == null) {
            developerStatus.setValue(AppSecData.VulnerabilityStatus.NOT_SET);
        }

        String id = vulnerabilityDTO.getIdentifier();
        AppSecData data = AppSecService.getInstance().getData();

        AppSecData.VulnerabilityAssessment vulnerability = data
                .getVulnerabilities().get(id);
        if (vulnerability == null) {
            vulnerability = new AppSecData.VulnerabilityAssessment();
        }
        vulnerability.setId(id);
        vulnerability.setStatus(developerStatus.getValue());
        vulnerability.setDeveloperAnalysis(developerAnalysis.getValue());

        Instant now = Instant.now();
        vulnerability.setUpdated(now);

        // Save new values through service
        data.getVulnerabilities().put(id, vulnerability);
        AppSecService.getInstance().setData(data);

        // Update UI
        lastUpdated.setText(
                "Last updated: " + dateTimeFormatter.format(Date.from(now)));
        saveButton.setEnabled(false);
    }
}
