/*-
 * Copyright (C) 2023 Vaadin Ltd
 *
 * This program is available under Vaadin Commercial License and Service Terms.
 *
 *
 * See <https://vaadin.com/commercial-license-and-service-terms> for the full license.
 */
package com.vaadin.appsec.backend.model.dto;

import javax.validation.constraints.NotNull;
import java.io.Serializable;
import java.time.Instant;
import java.util.Date;
import java.util.Objects;
import java.util.Set;

import com.vaadin.appsec.backend.model.AppSecData;
import com.vaadin.appsec.backend.model.analysis.AffectedVersion;

/**
 * DTO for a vulnerability instance, used in the UI.
 */
public class Vulnerability implements Serializable {

    private final String identifier;
    private Dependency dependency;
    private String patchedVersion;
    private AffectedVersion affectedVersion;
    private AppSecData.VulnerabilityStatus developerStatus;
    private String developerAnalysis;
    private Instant developerUpdated;
    private Date datePublished;
    private String details;
    private Set<String> referenceUrls;

    /**
     * Instantiates a new Vulnerability dto.
     *
     * @param identifier
     *            the identifier
     */
    public Vulnerability(@NotNull String identifier) {
        this.identifier = identifier;
    }

    /**
     * Gets identifier.
     *
     * @return the identifier
     */
    public String getIdentifier() {
        return identifier;
    }

    /**
     * Gets dependency.
     *
     * @return the dependency
     */
    public Dependency getDependency() {
        return dependency;
    }

    /**
     * Sets dependency.
     *
     * @param dependency
     *            the dependency
     */
    public void setDependency(Dependency dependency) {
        this.dependency = dependency;
    }

    /**
     * Gets patched version.
     *
     * @return the patched version
     */
    public String getPatchedVersion() {
        return patchedVersion;
    }

    /**
     * Sets patched version.
     *
     * @param patchedVersion
     *            the patched version
     */
    public void setPatchedVersion(String patchedVersion) {
        this.patchedVersion = patchedVersion;
    }

    /**
     * Gets severity level.
     *
     * @return the severity level
     */
    public SeverityLevel getSeverityLevel() {
        return dependency.getSeverityLevel();
    }

    /**
     * Gets CVSS string.
     *
     * @return the CVSS string
     */
    public String getCvssString() {
        return dependency.getCvssString();
    }

    /**
     * Gets risk score.
     *
     * @return the risk score
     */
    public Double getRiskScore() {
        return dependency.getRiskScore();
    }

    /**
     * Gets affected version.
     *
     * @return the affected version
     */
    public AffectedVersion getAffectedVersion() {
        return affectedVersion;
    }

    /**
     * Sets affected version.
     *
     * @param affectedVersion
     *            the affected version
     */
    public void setAffectedVersion(AffectedVersion affectedVersion) {
        this.affectedVersion = affectedVersion;
    }

    /**
     * Gets developer analysis.
     *
     * @return the developer analysis
     */
    public String getDeveloperAnalysis() {
        return developerAnalysis;
    }

    /**
     * Sets developer analysis.
     *
     * @param developerAnalysis
     *            the developer analysis
     */
    public void setDeveloperAnalysis(String developerAnalysis) {
        this.developerAnalysis = developerAnalysis;
    }

    public Date getDatePublished() {
        return datePublished;
    }

    public void setDatePublished(Date datePublished) {
        this.datePublished = datePublished;
    }

    public String getDetails() {
        return details;
    }

    public void setDetails(String details) {
        this.details = details;
    }

    public Set<String> getReferenceUrls() {
        return referenceUrls;
    }

    public void setReferenceUrls(Set<String> referenceUrls) {
        this.referenceUrls = referenceUrls;
    }

    public AppSecData.VulnerabilityStatus getDeveloperStatus() {
        return developerStatus;
    }

    public void setDeveloperStatus(
            AppSecData.VulnerabilityStatus developerStatus) {
        this.developerStatus = developerStatus;
    }

    public Instant getDeveloperUpdated() {
        return developerUpdated;
    }

    public void setDeveloperUpdated(Instant developerUpdated) {
        this.developerUpdated = developerUpdated;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o)
            return true;
        if (o == null || getClass() != o.getClass())
            return false;
        Vulnerability that = (Vulnerability) o;
        return identifier.equals(that.identifier)
                && Objects.equals(dependency, that.dependency);
    }

    @Override
    public int hashCode() {
        return Objects.hash(identifier, dependency);
    }

    @Override
    public String toString() {
        return "VulnerabilityDTO{" + "identifier='" + identifier + '\''
                + ", dependency=" + dependency + ", datePublished="
                + datePublished + '}';
    }
}
