/*-
 * Copyright (C) 2023 Vaadin Ltd
 *
 * This program is available under Vaadin Commercial License and Service Terms.
 *
 *
 * See <https://vaadin.com/commercial-license-and-service-terms> for the full license.
 */
package com.vaadin.appsec.backend.model.analysis;

import java.io.IOException;
import java.io.Serializable;
import java.util.HashMap;
import java.util.Map;

import com.fasterxml.jackson.core.JsonParser;
import com.fasterxml.jackson.databind.DeserializationContext;
import com.fasterxml.jackson.databind.JsonDeserializer;
import com.fasterxml.jackson.databind.annotation.JsonDeserialize;

/**
 * The container of the vulnerability analysis coming from the Vaadin Security
 * Team.
 */
public class VulnerabilityAnalysis implements Serializable {

    static class MapDeserializer
            extends JsonDeserializer<VulnerabilityDetails> {

        @Override
        public VulnerabilityDetails deserialize(JsonParser p,
                DeserializationContext ctxt) throws IOException {
            String id = ctxt.getParser().getCurrentName();
            VulnerabilityDetails vulnerability = p
                    .readValueAs(VulnerabilityDetails.class);
            vulnerability.setId(id);
            return vulnerability;
        }
    }

    @JsonDeserialize(contentUsing = MapDeserializer.class)
    private Map<String, VulnerabilityDetails> vulnerabilities = new HashMap<>();

    /**
     * Gets the list of known vulnerabilities the Vaadin Security Team has
     * provided assessments for.
     *
     * @return a map with information about vulnerability assessments
     */
    public Map<String, VulnerabilityDetails> getVulnerabilities() {
        return vulnerabilities;
    }

    void setVulnerabilities(Map<String, VulnerabilityDetails> vulnerabilities) {
        this.vulnerabilities = vulnerabilities;
    }

    @Override
    public String toString() {
        return "VulnerabilityAnalysis{" + "vulnerabilities=" + vulnerabilities
                + '}';
    }
}
