/*-
 * Copyright (C) 2023 Vaadin Ltd
 *
 * This program is available under Vaadin Commercial License and Service Terms.
 *
 *
 * See <https://vaadin.com/commercial-license-and-service-terms> for the full license.
 */
package com.vaadin.appsec.backend;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.HttpURLConnection;
import java.net.SocketTimeoutException;
import java.net.URI;
import java.net.URL;

import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.databind.ObjectMapper;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.vaadin.appsec.backend.model.osv.request.QueryBatchRequestPayload;
import com.vaadin.appsec.backend.model.osv.response.OpenSourceVulnerability;
import com.vaadin.appsec.backend.model.osv.response.QueryBatchResponse;

/**
 * Client implementation for fetching data from osv.dev
 */
class OpenSourceVulnerabilityClient {

    private static final Logger LOGGER = LoggerFactory
            .getLogger(OpenSourceVulnerabilityClient.class);

    private static final String QUERY_BATCH_URL = "https://api.osv.dev/v1/querybatch";
    private static final String VULNERABILITY_URL = "https://api.osv.dev/v1/vulns/";

    private final ObjectMapper mapper = new ObjectMapper()
            .setSerializationInclusion(JsonInclude.Include.NON_NULL);

    private final RateLimiter rateLimiter;

    OpenSourceVulnerabilityClient(int ratePerSecond) {
        rateLimiter = new RateLimiter(ratePerSecond);
    }

    /**
     * Do a batch query to the OSV API to check vulnerabilities of multiple
     * software packages in a single request.
     *
     * @param payload
     *            the payload to send to OSV API server with all packages to
     *            check.
     * @return a response with a list of vulnerabilities.
     * @throws AppSecException
     *             if there is a connection issue with the OSV API server.
     */
    QueryBatchResponse queryBatch(QueryBatchRequestPayload payload)
            throws AppSecException {
        LOGGER.debug("Performing a batch query to the OSV API...");
        HttpURLConnection conn = createHttpURLConnection(getQueryBatchUrl(),
                "POST");

        rateLimit();

        try (OutputStream outputStream = conn.getOutputStream()) {
            mapper.writeValue(outputStream, payload);
        } catch (IOException e) {
            throw new AppSecException(
                    "Failed to write OSV API query batch request payload", e);
        }

        try (InputStream inputStream = conn.getInputStream()) {
            return mapper.readValue(inputStream, QueryBatchResponse.class);
        } catch (IOException e) {
            throw new AppSecException(
                    "Failed to read OSV API query batch response", e);
        }
    }

    /**
     * Query the OSV API server for information about a specific vulnerability.
     *
     * @param vulnerabilityId
     *            the id of the vulnerability.
     * @return the information regarding an identified vulnerability.
     * @throws AppSecException
     *             if there's a connection issue with the OSV API server.
     */
    OpenSourceVulnerability queryVulnerability(String vulnerabilityId)
            throws AppSecException {
        LOGGER.debug("Performing a vulnerability query to the OSV API...");
        String urlWithId = getVulnerabilityUrl().concat(vulnerabilityId);
        HttpURLConnection conn = createHttpURLConnection(urlWithId, "GET");

        rateLimit();

        try (InputStream inputStream = conn.getInputStream()) {
            return mapper.readValue(inputStream, OpenSourceVulnerability.class);
        } catch (IOException e) {
            throw new AppSecException("Failed to read OSV API query response",
                    e);
        }
    }

    private HttpURLConnection createHttpURLConnection(String urlStr,
            String requestMethod) {
        HttpURLConnection connection;
        try {
            URL url = URI.create(urlStr).toURL();
            connection = (HttpURLConnection) url.openConnection();
            connection.setRequestMethod(requestMethod);
            connection.setRequestProperty("Content-Type", "application/json");
            connection.setRequestProperty("Accept", "application/json");
            connection.setDoOutput(true);
            connection.setConnectTimeout(5000);
        } catch (SocketTimeoutException e) {
            throw new AppSecException(
                    "Connection to OSV API server has timed out", e);
        } catch (IOException e) {
            throw new AppSecException(
                    "Failed to create connection to OSV API with URL: "
                            + urlStr,
                    e);
        }
        return connection;
    }

    String getQueryBatchUrl() {
        return QUERY_BATCH_URL;
    }

    String getVulnerabilityUrl() {
        return VULNERABILITY_URL;
    }

    private void rateLimit() {
        try {
            rateLimiter.limit();
        } catch (InterruptedException e) {
            throw new AppSecException(
                    "OpenSourceVulnerabilityClient rate limit exceeded", e);
        }
    }
}
