/*-
 * Copyright (C) 2023 Vaadin Ltd
 *
 * This program is available under Vaadin Commercial License and Service Terms.
 *
 *
 * See <https://vaadin.com/commercial-license-and-service-terms> for the full license.
 */

package com.vaadin.appsec.backend;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import org.cyclonedx.model.Component;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.vaadin.appsec.backend.model.osv.request.QueryBatchRequestPayload;
import com.vaadin.appsec.backend.model.osv.request.QueryRequestPayload;
import com.vaadin.appsec.backend.model.osv.response.OpenSourceVulnerability;
import com.vaadin.appsec.backend.model.osv.response.Package;
import com.vaadin.appsec.backend.model.osv.response.QueryBatchResponse;
import com.vaadin.appsec.backend.model.osv.response.VulnerabilityId;
import com.vaadin.appsec.backend.model.osv.response.VulnerabilityIdArray;

/**
 * Service for fetching and processing vulnerabilities using
 * {@link OpenSourceVulnerabilityClient}
 */
class OpenSourceVulnerabilityService {

    private static final Logger LOGGER = LoggerFactory
            .getLogger(OpenSourceVulnerabilityService.class);

    private static final int COMPONENT_BATCH_SIZE = 1000;

    private final OpenSourceVulnerabilityClient osvClient;

    OpenSourceVulnerabilityService(int ratePerSecond) {
        osvClient = new OpenSourceVulnerabilityClient(ratePerSecond);
    }

    /**
     * Returns fetched vulnerabilities for given list of Bom components
     *
     * @param components
     *            the list of BOM components
     * @return the list of vulnerabilities
     */
    List<OpenSourceVulnerability> getVulnerabilities(
            List<Component> components) {
        return batchComponents(components)
                .map(this::createQueryBatchRequestPayload)
                .map(osvClient::queryBatch).map(QueryBatchResponse::getResults)
                .flatMap(Arrays::stream)
                .map(VulnerabilityIdArray::getVulnerabilityIds)
                .filter(Objects::nonNull)
                .peek(vuln -> LOGGER
                        .debug("Collecting data for " + Arrays.toString(vuln)))
                .flatMap(Arrays::stream).map(VulnerabilityId::getId)
                .map(osvClient::queryVulnerability)
                .collect(Collectors.toList());
    }

    private Stream<List<Component>> batchComponents(
            List<Component> components) {
        List<List<Component>> batches = new ArrayList<>();
        for (int i = 0; i < components.size(); i += COMPONENT_BATCH_SIZE) {
            batches.add(components.subList(i,
                    Math.min(i + COMPONENT_BATCH_SIZE, components.size())));
        }
        return batches.stream();
    }

    private QueryBatchRequestPayload createQueryBatchRequestPayload(
            List<Component> components) {
        return new QueryBatchRequestPayload(
                components.stream().map(this::createQueryRequestPayload)
                        .toArray(QueryRequestPayload[]::new));
    }

    private QueryRequestPayload createQueryRequestPayload(Component component) {
        return new QueryRequestPayload(new Package(component.getPurl()));
    }
}
