/*-
 * Copyright (C) 2023 Vaadin Ltd
 *
 * This program is available under Vaadin Commercial License and Service Terms.
 *
 *
 * See <https://vaadin.com/commercial-license-and-service-terms> for the full license.
 */

package com.vaadin.appsec.backend.model.osv.response;

import java.util.Date;
import java.util.List;
import java.util.Objects;

import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.fasterxml.jackson.annotation.JsonPropertyOrder;

/**
 * Open Source Vulnerability model for describing a vulnerability in an open
 * source package.
 *
 * @see <a href=
 *      "https://github.com/ossf/osv-schema/blob/main/validation/schema.json">OSV-Schema</a>
 */
@JsonInclude(JsonInclude.Include.NON_NULL)
@JsonPropertyOrder({ "schema_version", "id", "modified", "published",
        "withdrawn", "aliases", "related", "summary", "details", "severity",
        "affected", "references", "credits", "database_specific" })
public class OpenSourceVulnerability {

    @JsonProperty("schema_version")
    private String schemaVersion;
    @JsonProperty("id")
    private String id;
    @JsonProperty("modified")
    private Date modified;
    @JsonProperty("published")
    private Date published;
    @JsonProperty("withdrawn")
    private Date withdrawn;
    @JsonProperty("aliases")
    private List<String> aliases;
    @JsonProperty("related")
    private List<String> related;
    @JsonProperty("summary")
    private String summary;
    @JsonProperty("details")
    private String details;
    @JsonProperty("severity")
    private List<Severity> severity;
    @JsonProperty("affected")
    private List<Affected> affected;
    @JsonProperty("references")
    private List<Reference> references;
    @JsonProperty("credits")
    private List<Credit> credits;
    @JsonProperty("database_specific")
    private DatabaseSpecific databaseSpecific;

    /**
     * No args constructor for use in serialization.
     */
    public OpenSourceVulnerability() {
    }

    /**
     * Instantiates a new open source vulnerability.
     *
     * @param schemaVersion
     *            the schema version
     * @param id
     *            the id
     * @param modified
     *            the modified
     * @param published
     *            the published
     * @param withdrawn
     *            the withdrawn
     * @param aliases
     *            the aliases
     * @param related
     *            the related
     * @param summary
     *            the summary
     * @param details
     *            the details
     * @param severity
     *            the severity
     * @param affected
     *            the affected
     * @param references
     *            the references
     * @param credits
     *            the credits
     * @param databaseSpecific
     *            the database specific
     */
    public OpenSourceVulnerability(String schemaVersion, String id,
            Date modified, Date published, Date withdrawn, List<String> aliases,
            List<String> related, String summary, String details,
            List<Severity> severity, List<Affected> affected,
            List<Reference> references, List<Credit> credits,
            DatabaseSpecific databaseSpecific) {
        this.schemaVersion = schemaVersion;
        this.id = id;
        this.modified = modified;
        this.published = published;
        this.withdrawn = withdrawn;
        this.aliases = aliases;
        this.related = related;
        this.summary = summary;
        this.details = details;
        this.severity = severity;
        this.affected = affected;
        this.references = references;
        this.credits = credits;
        this.databaseSpecific = databaseSpecific;
    }

    /**
     * Gets the schema version.
     *
     * @return the schema version
     */
    public String getSchemaVersion() {
        return schemaVersion;
    }

    /**
     * Sets the schema version.
     *
     * @param schema_version
     *            the new schema version
     */
    public void setSchemaVersion(String schema_version) {
        this.schemaVersion = schema_version;
    }

    /**
     * Gets the id.
     *
     * @return the id
     */
    public String getId() {
        return id;
    }

    /**
     * Sets the id.
     *
     * @param id
     *            the new id
     */
    public void setId(String id) {
        this.id = id;
    }

    /**
     * Gets the modified.
     *
     * @return the modified
     */
    public Date getModified() {
        return modified;
    }

    /**
     * Sets the modified.
     *
     * @param modified
     *            the new modified
     */
    public void setModified(Date modified) {
        this.modified = modified;
    }

    /**
     * Gets the published.
     *
     * @return the published
     */
    public Date getPublished() {
        return published;
    }

    /**
     * Sets the published.
     *
     * @param published
     *            the new published
     */
    public void setPublished(Date published) {
        this.published = published;
    }

    /**
     * Gets the withdrawn.
     *
     * @return the withdrawn
     */
    public Date getWithdrawn() {
        return withdrawn;
    }

    /**
     * Sets the withdrawn.
     *
     * @param withdrawn
     *            the new withdrawn
     */
    public void setWithdrawn(Date withdrawn) {
        this.withdrawn = withdrawn;
    }

    /**
     * Gets the aliases.
     *
     * @return the aliases
     */
    public List<String> getAliases() {
        return aliases;
    }

    /**
     * Sets the aliases.
     *
     * @param aliases
     *            the new aliases
     */
    public void setAliases(List<String> aliases) {
        this.aliases = aliases;
    }

    /**
     * Gets the related.
     *
     * @return the related
     */
    public List<String> getRelated() {
        return related;
    }

    /**
     * Sets the related.
     *
     * @param related
     *            the new related
     */
    public void setRelated(List<String> related) {
        this.related = related;
    }

    /**
     * Gets the summary.
     *
     * @return the summary
     */
    public String getSummary() {
        return summary;
    }

    /**
     * Sets the summary.
     *
     * @param summary
     *            the new summary
     */
    public void setSummary(String summary) {
        this.summary = summary;
    }

    /**
     * Gets the details.
     *
     * @return the details
     */
    public String getDetails() {
        return details;
    }

    /**
     * Sets the details.
     *
     * @param details
     *            the new details
     */
    public void setDetails(String details) {
        this.details = details;
    }

    /**
     * Gets the severity.
     *
     * @return the severity
     */
    public List<Severity> getSeverity() {
        return severity;
    }

    /**
     * Sets the severity.
     *
     * @param severity
     *            the new severity
     */
    public void setSeverity(List<Severity> severity) {
        this.severity = severity;
    }

    /**
     * Gets the affected.
     *
     * @return the affected
     */
    public List<Affected> getAffected() {
        return affected;
    }

    /**
     * Sets the affected.
     *
     * @param affected
     *            the new affected
     */
    public void setAffected(List<Affected> affected) {
        this.affected = affected;
    }

    /**
     * Gets the references.
     *
     * @return the references
     */
    public List<Reference> getReferences() {
        return references;
    }

    /**
     * Sets the references.
     *
     * @param references
     *            the new references
     */
    public void setReferences(List<Reference> references) {
        this.references = references;
    }

    /**
     * Gets the credits.
     *
     * @return the credits
     */
    public List<Credit> getCredits() {
        return credits;
    }

    /**
     * Sets the credits.
     *
     * @param credits
     *            the new credits
     */
    public void setCredits(List<Credit> credits) {
        this.credits = credits;
    }

    /**
     * Database specific.
     *
     * @return the database specific
     */
    public DatabaseSpecific databaseSpecific() {
        return databaseSpecific;
    }

    /**
     * Sets the database specific.
     *
     * @param databaseSpecific
     *            the new database specific
     */
    public void setDatabaseSpecific(DatabaseSpecific databaseSpecific) {
        this.databaseSpecific = databaseSpecific;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        OpenSourceVulnerability that = (OpenSourceVulnerability) o;
        return Objects.equals(id, that.id);
    }

    @Override
    public int hashCode() {
        return Objects.hash(id);
    }

    @Override
    public String toString() {
        return "OpenSourceVulnerability{" + "schemaVersion='" + schemaVersion
                + '\'' + ", id='" + id + '\'' + ", modified=" + modified
                + ", published=" + published + ", withdrawn=" + withdrawn
                + ", aliases=" + aliases + ", severity=" + severity
                + ", affected=" + affected + '}';
    }
}
