/*-
 * Copyright (C) 2023 Vaadin Ltd
 *
 * This program is available under Vaadin Commercial License and Service Terms.
 *
 *
 * See <https://vaadin.com/commercial-license-and-service-terms> for the full license.
 */
package com.vaadin.appsec.backend.model.analysis;

import java.io.IOException;
import java.io.Serializable;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;

import com.fasterxml.jackson.annotation.JsonIgnore;
import com.fasterxml.jackson.core.JsonParser;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.DeserializationContext;
import com.fasterxml.jackson.databind.JsonDeserializer;
import com.fasterxml.jackson.databind.annotation.JsonDeserialize;

/**
 * Container of information about a single vulnerability. It provides the
 * dependency that brings the vulnerability and assessment for Vaadin modules
 * depending on that dependency in the maintained versions.
 */
public class VulnerabilityDetails implements Serializable {

    static class MapDeserializer extends JsonDeserializer<Assessment> {

        @Override
        public Assessment deserialize(JsonParser p, DeserializationContext ctxt)
                throws IOException, JsonProcessingException {
            String name = ctxt.getParser().getCurrentName();
            Assessment assessment = p.readValueAs(Assessment.class);
            assessment.setName(name);
            return assessment;
        }
    }

    @JsonIgnore
    private String id;

    private Dependency dependency;

    @JsonDeserialize(contentUsing = MapDeserializer.class)
    private Map<String, Assessment> assessments = new HashMap<>();

    /**
     * Gets the vulnerability identifier.
     *
     * @return the vulnerability identifier
     */
    public String getId() {
        return id;
    }

    void setId(String id) {
        this.id = id;
    }

    /**
     * Gets the dependency informations.
     *
     * @return the dependency informations
     */
    public Dependency getDependency() {
        return dependency;
    }

    void setDependency(Dependency dependency) {
        this.dependency = dependency;
    }

    /**
     * Gets a map of assessments by the Vaadin Security Team.
     *
     * @return the assessments
     */
    public Map<String, Assessment> getAssessments() {
        return assessments;
    }

    void setAssessments(Map<String, Assessment> assessments) {
        this.assessments = assessments;
    }

    @Override
    public int hashCode() {
        return Objects.hash(id);
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (!(obj instanceof VulnerabilityDetails)) {
            return false;
        }
        VulnerabilityDetails other = (VulnerabilityDetails) obj;
        return Objects.equals(id, other.id);
    }

    @Override
    public String toString() {
        return "Vulnerability{" + "id='" + id + '\'' + ", dependency="
                + dependency + ", assessments=" + assessments + '}';
    }
}
