/*-
 * Copyright (C) 2023 Vaadin Ltd
 *
 * This program is available under Vaadin Commercial License and Service Terms.
 *
 *
 * See <https://vaadin.com/commercial-license-and-service-terms> for the full license.
 */

package com.vaadin.appsec.backend;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;

import org.cyclonedx.model.Component;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.vaadin.appsec.backend.model.osv.response.Ecosystem;
import com.vaadin.appsec.backend.model.osv.response.OpenSourceVulnerability;

/**
 * Provides means to store and fetch vulnerabilities from a static instance.
 */
class VulnerabilityStore {

    private static final Logger LOGGER = LoggerFactory
            .getLogger(VulnerabilityStore.class);

    private final List<OpenSourceVulnerability> vulnerabilities = new ArrayList<>();

    private final OpenSourceVulnerabilityService osvService;

    private final BillOfMaterialsStore bomStore;

    VulnerabilityStore(OpenSourceVulnerabilityService osvService,
            BillOfMaterialsStore bomStore) {
        this.osvService = osvService;
        this.bomStore = bomStore;
    }

    /**
     * Initializes the store with the given vulnerabilities.
     *
     * @param vulnerabilities
     *            list of vulnerabilities to store
     */
    void init(List<OpenSourceVulnerability> vulnerabilities) {
        this.vulnerabilities.clear();
        this.vulnerabilities.addAll(new HashSet<>(vulnerabilities));
    }

    /**
     * Returns all stored vulnerabilities from the store.
     *
     * @return list of stored vulnerabilities
     */
    List<OpenSourceVulnerability> getVulnerabilities() {
        return Collections.unmodifiableList(vulnerabilities);
    }

    /**
     * Refreshes vulnerability store from database.
     *
     * @return updated list of vulnerabilities
     * @throws AppSecException
     *             if there is a connection issue with the vulnerabilities'
     *             database.
     */
    List<OpenSourceVulnerability> refresh() throws AppSecException {
        List<Component> components = new ArrayList<>(
                bomStore.getBom(Ecosystem.MAVEN).getComponents());
        if (bomStore.getBom(Ecosystem.NPM) != null) {
            components.addAll(bomStore.getBom(Ecosystem.NPM).getComponents());
        }

        LOGGER.debug("Scan for vulnerabilities started...");
        List<OpenSourceVulnerability> vulnerabilities = osvService
                .getVulnerabilities(components);
        LOGGER.debug("Scan for vulnerabilities finished");

        init(vulnerabilities);
        LOGGER.debug("Vulnerability store refreshed with " + vulnerabilities);
        return getVulnerabilities();
    }
}
