/*
 * Copyright 2020, Stefan Uebe
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy of this software and associated
 * documentation files (the "Software"), to deal in the Software without restriction, including without limitation the
 * rights to use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies of the Software, and to
 * permit persons to whom the Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all copies or substantial portions
 * of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
 * WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR
 * COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR
 * OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */
package org.vaadin.stefan.fullcalendar;

import com.vaadin.flow.component.ComponentEvent;
import lombok.Getter;
import lombok.ToString;
import tools.jackson.databind.node.ObjectNode;

import java.time.LocalDate;
import java.util.Optional;

import static org.vaadin.stefan.fullcalendar.JsonUtils.parseClientSideDate;

/**
 * Basic event for view render events. Provides information about the shown timespan.
 * <br><br>
 * The values are always daybased, regardless of the current view.
 */
@Getter
@ToString
public abstract class ViewRenderEvent extends ComponentEvent<FullCalendar> {

    /**
     * The client side name of the view.
     */
    private final String viewName;

    /**
     * The current shown interval's start date.
     */
    private final LocalDate intervalStart;

    /**
     * The current shown interval's end date.
     */
    private final LocalDate intervalEnd;

    /**
     * The first visible date. In month-view, this value is often before
     * the 1st day of the month, because most months do not begin on the first
     * day-of-week.
     */
    private final LocalDate start;

    /**
     * The last visible date. In month-view, this value is often after
     * the last day of the month, because most months do not end on the last day of the week
     */
    private final LocalDate end;


    private final CalendarView calendarView;

    /**
     * Creates a new event using the given source and indicator whether the
     * event originated from the client side or the server side.
     *
     * @param source     the source component
     * @param fromClient <code>true</code> if the event originated from the client
     */
    public ViewRenderEvent(FullCalendar source, boolean fromClient, ObjectNode eventData) {
        super(source, fromClient);

        this.viewName = eventData.get("name").asString();

        this.calendarView = source.lookupViewByClientSideValue(viewName).orElse(null);

        this.intervalStart = parseClientSideDate(eventData.get("intervalStart").asString());
        this.intervalEnd = parseClientSideDate(eventData.get("intervalEnd").asString());
        this.start = parseClientSideDate(eventData.get("start").asString());
        this.end = parseClientSideDate(eventData.get("end").asString());
    }

    /**
     * Same as {@link #getViewName()}
     * @deprecated use {@link #getViewName()} instead
     * @return Client side name of the view
     */
    @Deprecated(forRemoval = true)
    public String getName() {
        return viewName;
    }

    /**
     * The calendar view of this event. Empty, if the view name could not be matched with one of the predefined
     * views (e.g. in case of a custom view).
     * @return calendar view
     */
    public Optional<CalendarView> getCalendarView() {
        return Optional.ofNullable(calendarView);
    }
}
