/*
 * Copyright 2025 Vaadin Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not use this file except
 * in compliance with the License. You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software distributed under the License
 * is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express
 * or implied. See the License for the specific language governing permissions and limitations under
 * the License.
 */

import {
	createHourlyView,
	createDailyView,
	createConfig,
	TimeUnits
} from '@sx-premium/resource-scheduler';

import {
	addEvent,
	createCommonCalendar,
	navigateCalendar,
	removeEvent,
	setView,
	setDate, 
	setFirstDayOfWeek,
	setLocale,
	setTimeZone,
	setViews,
	setDayBoundaries,
	setWeekOptions,
	setCalendars,
	setMinDate,
	setMaxDate,
	setMonthGridOptions,
	updateEvent,
	onUpdateRange
} from './vcf-schedule-x-base.js';

import {
	updateResourceSchedulerRange, 
	processConfiguration
} from './vcf-schedule-x-utils.js';

import { signal } from "@preact/signals";

import '@sx-premium/resource-scheduler/index.css';
import '@sx-premium/scheduling-assistant/index.css';

const resourceViewFactoryMap = {
	createHourlyView,
	createDailyView
};

const resourceViewNameMap = {
	createHourlyView: createHourlyView().name,
	createDailyView: createDailyView().name
};

window.vcfschedulexresourcescheduler = {
	create(container, viewsJson, configJson, calendarsJson, resourceConfigJson, schedulingAssistantJson, currentViewJson) {
		setTimeout(() => {
			const config = processConfiguration(configJson, resourceViewNameMap);
			const resourceConfig = createConfig();
			this._processResourceSchedulerConfig(resourceConfig, resourceConfigJson);
			
			// attach lazy loading callbacks
			// callback that runs when the user scrolls the daily view
		    resourceConfig.onLazyLoadMonth = (dates) => {
				if (!Array.isArray(dates) || dates.length === 0) return;

				const startPlainDate = dates[0];
				const start = startPlainDate.toPlainDateTime();
				let end;
				
				if (dates.length === 1) {
				  // only one date, extend to the end of that month
				  end = startPlainDate.with({ day: startPlainDate.daysInMonth });
				  end = end.toPlainDateTime(Temporal.PlainTime.from({ hour: 23, minute: 59, second: 59 }));	
				} else {
				  // the last visible date
				  const lastPlainDate = dates[dates.length - 1]
				  end = lastPlainDate.toPlainDateTime();
				}	
				
				updateResourceSchedulerRange(container, { start, end });
			};
		
			// callback that runs when the user scrolls the hourly view
		    resourceConfig.onLazyLoadDate = (dates) => {
				if (!Array.isArray(dates) || dates.length === 0) return;

				const startPlainDate = dates[0];
				const start = startPlainDate.toPlainDateTime();
				let end;
				
				if (dates.length === 1) {
				  // only one date, extend to end of that day
				  end = startPlainDate.with({ day: startPlainDate.daysInMonth });
				  end = end.toPlainDateTime(Temporal.PlainTime.from({ hour: 23, minute: 59, second: 59 }));
				} else {
				  // the last visible date
				  const lastPlainDate = dates[dates.length - 1]
				  end = lastPlainDate.toPlainDateTime();
				}	
				
				updateResourceSchedulerRange(container, { start, end });
		    };
			
			// get scheduling assistant configuration if available
			const schedulingAssistantConfig = this._processSchedulingAssistantConfig(schedulingAssistantJson, config);
			
			createCommonCalendar(container, resourceViewFactoryMap, config, calendarsJson, {
				viewsJson,
				resourceConfig,
				schedulingAssistantConfig
			});
			
			if(currentViewJson){
				this.setView(container,currentViewJson);
			}
		});
	},

	_processResourceSchedulerConfig(resourceConfig, resourceConfigJson) {
		const parsed = JSON.parse(resourceConfigJson);
		const timeUnits = new TimeUnits();

		this._assignIfExists(resourceConfig, parsed, 'hourWidth');
		this._assignIfExists(resourceConfig, parsed, 'dayWidth');
		this._assignIfExists(resourceConfig, parsed, 'resources', resources =>
			resources.map(resource => ({
				...resource,
				isOpen: signal(resource.isOpen)
			}))
		);
		this._assignIfExists(resourceConfig, parsed, 'resourceHeight');
		this._assignIfExists(resourceConfig, parsed, 'eventHeight');
		this._assignIfExists(resourceConfig, parsed, 'dragAndDrop');
		this._assignIfExists(resourceConfig, parsed, 'resize');
		this._assignIfExists(resourceConfig, parsed, 'infiniteScroll');
		this._assignIfExists(resourceConfig, parsed, 'initialHours', raw => {
			const [start, end] = this._parseInitialRange(raw);
			start = Temporal.PlainDateTime.from(start);
			end = Temporal.PlainDateTime.from(end);
			return timeUnits.getDayHoursBetween(start, end);
		});
		this._assignIfExists(resourceConfig, parsed, 'initialDays', raw => {
			const [start, end] = this._parseInitialRange(raw);
			start = Temporal.PlainDate.from(start);
			end = Temporal.PlainDate.from(end);
			return timeUnits.getDaysBetween(start, end);
		});
	},

	_parseInitialRange(value) {
		if (typeof value === 'string') {
			const [start, end] = value.split(',').map(s => s.trim());
			return [start, end];
		}
		return [];
	},

	_assignIfExists(target, source, key, transform = v => v) {
		const raw = source[key];
		if (raw !== undefined && raw !== '') {
			target[key].value = transform(raw);
		}
	},
	
	_processSchedulingAssistantConfig(schedulingAssistantJson, config) {
		const schedulingAssistantConfig = schedulingAssistantJson === "{}" ? null : JSON.parse(schedulingAssistantJson);
		const timeZone = config.timezone || 'UTC';
		if(schedulingAssistantConfig && schedulingAssistantConfig.initialStart) {
			const plainDateTime = Temporal.PlainDateTime.from(schedulingAssistantConfig.initialStart);
			const zoneDateTime = plainDateTime.toZonedDateTime(timeZone); 
			schedulingAssistantConfig.initialStart = zoneDateTime;
		}
		if(schedulingAssistantConfig && schedulingAssistantConfig.initialEnd) {
			const plainDateTime = Temporal.PlainDateTime.from(schedulingAssistantConfig.initialEnd);
		    const zoneDateTime = plainDateTime.toZonedDateTime(timeZone); 
			schedulingAssistantConfig.initialEnd = zoneDateTime;
		}
		return schedulingAssistantConfig
	},

	setView(container, view) {
		setView(container, view, resourceViewNameMap);
	},

	setDate(container, selectedDate) {
		setDate(container, selectedDate);
	},
	
	setFirstDayOfWeek(container, firstDayOfWeek) {
		setFirstDayOfWeek(container, firstDayOfWeek);
	},
	
	setLocale(container, locale) {
		setLocale(container, locale);
	},
	
	setTimeZone(container, timeZone) {
		setTimeZone(container, timeZone);
	},
	
	setViews(container, viewsJson) {
		setViews(container, viewsJson, resourceViewFactoryMap);
	},
	
	setDayBoundaries(container, dayBoundariesJson) {
		setDayBoundaries(container, dayBoundariesJson);
	},
	
	setWeekOptions(container, weekOptionsJson) {
		setWeekOptions(container, weekOptionsJson);
	},
	
	setCalendars(container, calendarsJson) {
		setCalendars(container, calendarsJson);
	},
	
	setMinDate(container, minDate){
		setMinDate(container, minDate);
	},
	
	setMaxDate(container, maxDate){
		setMaxDate(container, maxDate);
	},
	
	setMonthGridOptions(container, monthGridOptionsJson){
		setMonthGridOptions(container, monthGridOptionsJson);
	},
	
	addEvent(container, calendarEvent) {
		addEvent(container, calendarEvent);
	},
	
	removeEvent(container, calendarEventId) {
		removeEvent(container, calendarEventId);
	},
	
	updateEvent(container, calendarEvent) {
		updateEvent(container, calendarEvent);
	},

	onUpdateRange(container, events, start, end) {
		onUpdateRange(container, events, start, end);
	},
	
	navigateForwards(container) {
		navigateCalendar(container.calendar, 'forwards');
	},
	
	navigateBackwards(container) {
		navigateCalendar(container.calendar, 'backwards');
	},
};