package org.vaadin.addons.componentfactory;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Random;
import java.util.stream.Collectors;

import com.vaadin.flow.component.AttachEvent;
import com.vaadin.flow.component.ComponentEvent;
import com.vaadin.flow.component.ComponentEventListener;
import com.vaadin.flow.component.Composite;
import com.vaadin.flow.component.DomEvent;
import com.vaadin.flow.component.dependency.CssImport;
import com.vaadin.flow.component.dependency.JavaScript;
import com.vaadin.flow.component.dependency.StyleSheet;
import com.vaadin.flow.component.html.Div;
import com.vaadin.flow.data.binder.BeanPropertySet;
import com.vaadin.flow.data.binder.PropertySet;
import com.vaadin.flow.function.SerializableConsumer;
import com.vaadin.flow.shared.Registration;

import tools.jackson.databind.JsonNode;
import tools.jackson.databind.ObjectMapper;
import tools.jackson.databind.node.ArrayNode;
import tools.jackson.databind.node.ObjectNode;

@SuppressWarnings("serial")
@StyleSheet("context://c3/c3.min.css")
@StyleSheet("context://pivottable/dist/pivot.css")
@JavaScript("context://jquery/dist/jquery.min.js")
@JavaScript("context://jqueryui/jquery-ui.min.js")
@JavaScript("context://d3/build/d3.min.js")
@JavaScript("context://c3/c3.min.js")
@JavaScript("context://pivottable/dist/pivot.min.js")
@JavaScript("context://pivottable/dist/c3_renderers.min.js")
@JavaScript("context://pivottable/dist/export_renderers.min.js")
@JavaScript("context://tabletojson/lib/jquery.tabletojson.min.js")
@JavaScript("./pivot_connector.js")
@CssImport("./lumo-pivot.css")
/**
 * PivotTable is component based on pivottable.js. This component performs
 * pivoting of the dataset in the browser. Thus it is suitable for small
 * datasets which do not require lazy loading from the backend.
 */
public class PivotTable extends Composite<Div> {

    private String dataJson;
    private String optionsJson;
    private PivotMode pivotMode;
    private Random rand = new Random();
    private String id;
    private PivotOptions options;
    private PivotTableI18n i18n;

    /**
     * The mode, PivotMode.INTERACTIVE renders with Pivot UI.
     */
    public enum PivotMode {
        INTERACTIVE, NONINTERACTIVE
    }

    /**
     * Utility helpers for valid renderer strings.
     */
    public final class Renderer {
        public static final String TABLE = "Table";
        public static final String TABLE_BARCHART = "Table Barchart";
        public static final String TABLE_HEATMAP = "Heatmap";
        public static final String ROW_HEATMAP = "Row Heatmap";
        public static final String COL_HEATMAP = "Col Heatmap";
        public static final String HORIZONTAL_BAR_CHART = "Horizontal Bar Chart";
        public static final String HORIZONTAL_STACKED_BAR_CHART = "Horizontal Stacked Bar Chart";
        public static final String BAR_CHART = "Bar Chart";
        public static final String STACKED_BAR_CHART = "Stacked Bar Chart";
        public static final String LINE_CHART = "Line Chart";
        public static final String AREA_CHART = "Area Chart";
        public static final String SCATTER_CHART = "Scatter Chart";
        public static final String TSV_EXPORT = "TSV Export";
    }

    /**
     * Utility helpers for valid renderer strings.
     */
    public final class Aggregator {
        public static final String COUNT = "Count";
        public static final String COUNT_UNIQUE_VALUES = "Count Unique Values";
        public static final String LIST_UNIQUE_VALUES = "List Unique Values";
        public static final String SUM = "Sum";
        public static final String INTEGER_SUM = "Integer Sum";
        public static final String AVERAGE = "Average";
        public static final String MEDIAN = "Median";
        public static final String SAMPLE_VARIANCE = "Sample Variance";
        public static final String SAMPLE_STANDARD_DEVIATION = "Sample Standard Deviation";
        public static final String MINIMUM = "Minimum";
        public static final String MAXIMUM = "Maximum";
        public static final String FIRST = "First";
        public static final String LAST = "Last";
        public static final String SUM_OVER_SUM = "Sum over Sum";
        public static final String UPPER_BOUND = "80% Upper Bound";
        public static final String LOWER_BOUND = "80% Lower Bound";
        public static final String SUM_FRACTION_OF_TOTAL = "Sum as Fraction of Total";
        public static final String SUM_FRACTION_OF_ROWS = "Sum as Fraction of Rows";
        public static final String SUM_FRACTION_OF_COLS = "Sum as Fraction of Columns";
        public static final String COUNT_FRACTION_OF_TOTAL = "Count as Fraction of Total";
        public static final String COUNT_FRACTION_OF_ROWS = "Count as Fraction of Rows";
        public static final String COUNT_FRACTION_OF_COLS = "Count as Fraction of Columns";
    }

    /**
     * Options for PivotTable
     */
    public static class PivotOptions implements Serializable {
        List<String> cols;
        List<String> rows;
        List<String> disabledRerenders;
        String renderer;
        String aggregator;
        String column;
        boolean charts;
        boolean fieldsDisabled;

        public PivotOptions() {
        }

        /**
         * Set default columns for the pivot
         * 
         * @param cols
         *            Column identifiers
         */
        public void setCols(String... cols) {
            this.cols = Arrays.asList(cols);
        }

        /**
         * Set default rows for the pivot.
         * 
         * @param rows
         *            Row identifiers
         */
        public void setRows(String... rows) {
            this.rows = Arrays.asList(rows);
        }

        /**
         * Set disabled renderers.
         *
         * @see Renderer
         * 
         * @param renderers
         *            Renderers to disable.
         */
        public void setDisabledRenderers(String... renderers) {
            this.disabledRerenders = Arrays.asList(renderers);
        }

        /**
         * Set the default renderer.
         *
         * @see Renderer
         *
         * @param renderer
         *            The renderer name.
         */
        public void setRenderer(String renderer) {
            this.renderer = renderer;
        }

        /**
         * Set the default aggregator.
         *
         * @see Aggregator
         *
         * @param aggregator
         *            The aggregator name.
         * @param column
         *            The column name. Can be null.
         */
        public void setAggregator(String aggregator, String column) {
            this.aggregator = aggregator;
            this.column = column;
        }

        /**
         * When false fields cannot be rearranged.
         * 
         * @param fieldsDisabled
         *            Boolean value.
         */
        public void setFieldsDisabled(boolean fieldsDisabled) {
            this.fieldsDisabled = fieldsDisabled;
        }

        /**
         * Enable embbeded charts.
         * 
         * @param charts
         *            true for charts enabled.
         */
        public void setCharts(boolean charts) {
            this.charts = charts;
        }

        String toJson() {
            ObjectMapper mapper = PivotTable.MAPPER;
            ObjectNode object = mapper.createObjectNode();
            if (cols != null) {
                ArrayNode colsArray = object.putArray("cols");
                cols.forEach(colsArray::add);
            }
            if (rows != null) {
                ArrayNode rowsArray = object.putArray("rows");
                rows.forEach(rowsArray::add);
            }
            return object.toString();
        }
    }

    /**
     * Abastract base class for Pivot data models.
     */
    public static abstract class AbstractPivotData implements Serializable {

        LinkedHashMap<String, Class<?>> columns = new LinkedHashMap<>();
        List<Map<String, Object>> rows = new ArrayList<>();

        /**
         * Add column.
         * 
         * @param name
         *            Name of the column, unique.
         * @param type
         *            Data type used in the column
         */
        public void addColumn(String name, Class<?> type) {
            if (type.isAssignableFrom(Boolean.class)
                    || type.isAssignableFrom(Double.class)
                    || type.isAssignableFrom(Integer.class)
                    || type.isAssignableFrom(String.class)) {
                columns.put(name, type);
            } else {
                columns.put(name, String.class);
            }
        }

        /**
         * Add row from the map.
         * 
         * @param row
         *            Map of column key data object pairs.
         */
        public void addRow(Map<String, Object> row) {
            assert row.keySet().stream().allMatch(key -> columns.containsKey(
                    key)) : "Column key missing from configured columns.";
            rows.add(row);
        }

        String toJson() {
            ObjectMapper mapper = PivotTable.MAPPER;
            ArrayNode array = mapper.createArrayNode();
            rows.forEach(row -> {
                ObjectNode obj = mapper.createObjectNode();
                columns.forEach((name, type) -> {
                    Object value = row.get(name);
                    if (value == null) {
                        obj.putNull(name);
                    } else if (type.isAssignableFrom(Boolean.class)) {
                        obj.put(name, (Boolean) value);
                    } else if (type.isAssignableFrom(Double.class)) {
                        obj.put(name, (Double) value);
                    } else if (type.isAssignableFrom(Integer.class)) {
                        // original logic converted Integer to Double
                        obj.put(name, Double.valueOf((Integer) value));
                    } else if (type.isAssignableFrom(String.class)) {
                        obj.put(name, value.toString());
                    } else {
                        obj.put(name, value.toString());
                    }
                });
                array.add(obj);
            });
            return array.toString();
        }
    }

    /**
     * Pivot dataa model that auto creates based on list of beans using
     * introspection.
     * <p>
     * Note: Bean properties need to be compatible with Integer, Double, Boolean
     * to be considered either number or boolean on the client side Other types
     * are converted to String using object.toString.
     */
    public static class BeanPivotData<T> extends AbstractPivotData {

        PropertySet<T> propertySet;

        public BeanPivotData(Class<T> beanType, Collection<T> data) {
            propertySet = BeanPropertySet.get(beanType);
            propertySet.getProperties()
                    .filter(property -> !property.isSubProperty())
                    .sorted((prop1, prop2) -> prop1.getName()
                            .compareTo(prop2.getName()))
                    .forEach(prop -> addColumn(prop.getName(), prop.getType()));
            data.forEach(item -> {
                HashMap<String, Object> map = new HashMap<>();
                propertySet.getProperties()
                        .filter(property -> !property.isSubProperty())
                        .sorted((prop1, prop2) -> prop1.getName()
                                .compareTo(prop2.getName()))
                        .forEach(prop -> map.put(prop.getName(),
                                prop.getGetter().apply(item)));
                addRow(map);
            });
        }
    }

    /**
     * Data model for PivotTable. Columns need to be configured first, then add
     * rows.
     */
    public static class PivotData extends AbstractPivotData {

        public PivotData() {
        }

        /**
         * Add a row of data objects.
         * 
         * @param datas
         *            Data objects in the same order as columns were added.
         */
        public void addRow(Object... datas) {
            if (datas.length != columns.size()) {
                throw new IllegalArgumentException(
                        "Number of datas do not match with number of columns.");
            }
            Map<String, Object> map = new HashMap<>();
            int i = 0;
            for (String key : columns.keySet()) {
                map.put(key, datas[i]);
                i++;
            }
            addRow(map);
        }
    }

    public static class JsonPivotData extends AbstractPivotData {
        private String json;

        public JsonPivotData(String json) {
            this.json = json;
        }

        @Override
        public String toJson() {
            return json;
        }
    }

    /**
     * Create PivotTable using PivotMode.NONINTERACTIVE and given data and
     * options.
     * 
     * @param pivotData
     *            PivotData
     * @param pivotOptions
     *            PivotOptioms
     */
    public PivotTable(PivotData pivotData, PivotOptions pivotOptions) {
        this(pivotData, pivotOptions, PivotMode.NONINTERACTIVE);
    }

    /**
     * Create PivotTable using given data and options.
     * 
     * @param pivotData
     *            PivotData
     * @param pivotOptions
     *            PivotOptioms
     * @param mode
     *            The mode, PivotMode.INTERACTIVE renders PivotTable with
     *            interactive UI.
     */
    public PivotTable(AbstractPivotData pivotData, PivotOptions pivotOptions,
            PivotMode mode) {
        this.pivotMode = mode;
        id = randomId(10);
        setId(id);
        String optionsArray = pivotOptions.toJson();
        this.options = pivotOptions;
        this.dataJson = pivotData.toJson();
        this.optionsJson = optionsArray;
    }

    @Override
    protected void onAttach(AttachEvent event) {
        super.onAttach(event);
        if (options.charts) {
            String cols = options.cols != null
                    ? options.cols.stream().collect(Collectors.joining(","))
                    : null;
            String rows = options.rows != null
                    ? options.rows.stream().collect(Collectors.joining(","))
                    : null;
            String disabledRenderers = options.disabledRerenders != null
                    ? options.disabledRerenders.stream()
                            .collect(Collectors.joining(","))
                    : null;
            event.getUI().getPage().executeJs(
                    "window.drawChartPivotUI($0, $1, $2, $3, $4, $5, $6, $7, $8, $9);",
                    id, dataJson, cols, rows, disabledRenderers,
                    options.renderer, options.aggregator, options.column,
                    options.fieldsDisabled, pivotMode != PivotMode.INTERACTIVE);
        } else {
            event.getUI().getPage().executeJs(
                    "window.drawPivotUI($0, $1, $2, $3, $4, $5, $6, $7);", id,
                    dataJson, optionsJson, options.renderer, options.aggregator,
                    options.column, options.fieldsDisabled,
                    pivotMode != PivotMode.INTERACTIVE);
        }
    }

    /**
     * Fecth the current table result as Json. The result will be empty array if
     * the renderer is not a Table type.
     * 
     * @param callback
     *            Lambda function to be executed, gets fetched JsonValue as
     *            parameter.
     */
    public void fetchResult(SerializableConsumer<JsonNode> callback) {
        Objects.requireNonNull(callback,
                "Url consumer callback should not be null.");
        getElement().executeJs("return window.getPivotTableResult($0);", id)
                .then(callback::accept);
    }

    private String randomId(int chars) {
        int limit = (int) (Math.pow(10, chars) - 1);
        String key = "" + rand.nextInt(limit);
        key = String.format("%" + chars + "s", key).replace(' ', '0');
        return "pivot-" + key;
    }

    @SuppressWarnings({ "unchecked", "rawtypes" })
    public Registration addPivotReftreshedListener(
            ComponentEventListener<PivotRefreshedEvent<PivotTable>> listener) {
        return addListener(PivotRefreshedEvent.class,
                (ComponentEventListener) listener);
    }

    @DomEvent("pivot-refreshed")
    public static class PivotRefreshedEvent<R extends PivotTable>
            extends ComponentEvent<PivotTable> {

        public PivotRefreshedEvent(PivotTable source, boolean fromClient) {
            super(source, fromClient);
        }

    }

    /**
     * Set new i18n object in use and send values to client.
     * 
     * @param i18n
     *            The PivotTableI18n object.
     */
    public void setI18n(PivotTableI18n i18n) {
        this.i18n = i18n;
        getElement().executeJs("window.setPivotTableI18n($0, $1);", id,
                i18n.toJson());
    }

    /**
     * Get the localization object.
     * 
     * @return PivotTableI18n
     */
    public PivotTableI18n getI18n() {
        return i18n;
    }

    /**
     * Localization object for the PivotTable
     */
    public static class PivotTableI18n implements Serializable {
        Map<String, String> texts;

        /**
         * Constructor. Use String constants from Aggregator and Renderer as
         * keys.
         * 
         * @param texts
         *            Map of texts.
         */
        public PivotTableI18n(Map<String, String> texts) {
            Objects.requireNonNull(texts,
                    "Map of localized texts cannot be null");
            this.texts = texts;
        }

        String toJson() {
            ObjectMapper mapper = PivotTable.MAPPER;
            ArrayNode textArray = mapper.createArrayNode();
            texts.forEach((key, text) -> {
                ObjectNode textObj = mapper.createObjectNode();
                textObj.put("key", key);
                textObj.put("text", text);
                textArray.add(textObj);
            });
            return textArray.toString();
        }
    }

    // Central ObjectMapper instance (Jackson 3)
    private static final ObjectMapper MAPPER = new ObjectMapper();
}
