import { ComboBox } from '@vaadin/combo-box';
import { ComboBoxPlaceholder } from '@vaadin/combo-box/src/vaadin-combo-box-placeholder.js';

class AutoSelectComboBoxElement extends ComboBox {
  constructor() {
    super();
    this.previousInputLabel = '';
  }

  static get properties() {
    return {
      /**
       * Used to indicate that the internal validation failed.
       * @protected
       */
      _invalidInternal: {
        type: Boolean
      },

      /**
       * Used to indicate that the external validation failed.
       */
      invalidExternal: {
        type: Boolean,
        observer: '_invalidExternalChanged'
      }
    };
  }

  static get is() {
    return 'vcf-auto-select-combo-box';
  }

  ready() {
    super.ready();
    this.allowCustomValue = true;
    if (this.value === '') {
      this.dirty = false;
    } else {
      this.dirty = true;
    }
    this.addEventListener('custom-value-set', this._onCustomValueSet);
    this.addEventListener('value-changed', this._onValueSet);
  }

  _onValueSet(e) {
    if (e.detail.value) {
      if (this.items && this.__getItemIndexByValue(this.items, e.detail.value) < 0) {
        this._invalidInternal = true;
      } else {
        this._invalidInternal = false;
        if (e.detail.value !== '') {
          this.dirty = true;
        }
      }
      this._updateInvalidState();
    }
  }

  _onCustomValueSet(e) {
    if (this.previousInputLabel === e.detail) {
      return;
    }
    this.previousInputLabel = e.detail;

    this.checkValidity();
  }

  _detectAndDispatchChange() {
    if (this.value !== this._lastCommittedValue) {
      this.dispatchEvent(new CustomEvent('change', { bubbles: true }));
      this._lastCommittedValue = this.value;
      // keep track of latest input label
      this.previousInputLabel = this._getItemLabel(this.selectedItem);
      this.dirty = true;
    }
    this.checkValidity();
  }

  _filteredItemsChanged(filteredItems, oldFilteredItems) {
    super._filteredItemsChanged(filteredItems, oldFilteredItems);
    if (this.filteredItems && this.filteredItems.length === 1) {
      this._focusedIndex = 0;
    }
  }

  _invalidExternalChanged() {
    this._updateInvalidState();
  }

  _onClearAction() {
    if (this.readonly) {
      return;
    }
    this.selectedItem = null;
    if (this.allowCustomValue) {
      this.value = '';
    }
    this._detectAndDispatchChange();
  }

  checkValidity() {
    let validity = super.checkValidity();
    if (!this.dirty && this.inputElement?.value === '' && this.required) {
      // if there are no changes (not dirty) and input is empty and element is required, it's valid
      validity = true;
    } else if (!validity && this.inputElement?.value === '' && !this.required) {
      // if input is empty and element is not required, it's valid
      validity = true;
    } else if (!validity && this._filteredItemsContainFilterOrInputValue()) {
      // Invalid value was fixed to a valid one (filter string/filter input value exists in filteredItems)
      validity = true;
    } else if (validity && !this.selectedItem && !this._filteredItemsContainFilterOrInputValue()) {
      // if no item is selected and filtered items don't contain filter or input value
      // then this field is invalid
      validity = false;
    }
    this._invalidInternal = !validity;
    this._updateInvalidState();
    return validity;
  }

  _filteredItemsContainFilterOrInputValue() {
    return this.filteredItems.some(item => item.label == this.filter || item.label == this.inputElement.value);
  }

  _updateInvalidState() {
    this.invalid = this._invalidInternal || this.invalidExternal;
  }

  _closeOrCommit() {
    // Double check that the combobox placeholder is not the focused item
    let focusedItem = null;
    if (this._focusedIndex == 0 && this.filteredItems.length == 1) {
      focusedItem = this.filteredItems[this._focusedIndex];
    }
    if (focusedItem != null && typeof focusedItem == 'object' && focusedItem instanceof ComboBoxPlaceholder) {
      this.close();
    } else {
      super._closeOrCommit();
    }
  }
}

customElements.define(AutoSelectComboBoxElement.is, AutoSelectComboBoxElement);
