/*
 * Decompiled with CFR 0.152.
 */
package com.wontlost.ckeditor.internal;

import com.wontlost.ckeditor.handler.UploadHandler;
import java.io.ByteArrayInputStream;
import java.util.ArrayList;
import java.util.Base64;
import java.util.Map;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionException;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.logging.Level;
import java.util.logging.Logger;

public class UploadManager {
    private static final Logger logger = Logger.getLogger(UploadManager.class.getName());
    private static final long DEFAULT_UPLOAD_TIMEOUT_SECONDS = 360L;
    private final UploadHandler uploadHandler;
    private final UploadHandler.UploadConfig uploadConfig;
    private final UploadResultCallback resultCallback;
    private final Map<String, UploadTask> activeTasks = new ConcurrentHashMap<String, UploadTask>();
    private final long uploadTimeoutSeconds;

    public UploadManager(UploadHandler uploadHandler, UploadHandler.UploadConfig uploadConfig, UploadResultCallback resultCallback) {
        this(uploadHandler, uploadConfig, resultCallback, 360L);
    }

    public UploadManager(UploadHandler uploadHandler, UploadHandler.UploadConfig uploadConfig, UploadResultCallback resultCallback, long uploadTimeoutSeconds) {
        this.uploadHandler = uploadHandler;
        this.uploadConfig = uploadConfig != null ? uploadConfig : new UploadHandler.UploadConfig();
        this.resultCallback = resultCallback;
        this.uploadTimeoutSeconds = uploadTimeoutSeconds > 0L ? uploadTimeoutSeconds : 0L;
    }

    public void handleUpload(String uploadId, String fileName, String mimeType, String base64Data) {
        CompletableFuture<UploadHandler.UploadResult> future;
        byte[] fileData;
        logger.log(Level.FINE, "Starting upload: id={0}, file={1}, type={2}, dataLength={3}", new Object[]{uploadId, fileName, mimeType, base64Data != null ? base64Data.length() : 0});
        if (this.uploadHandler == null) {
            logger.log(Level.WARNING, "Upload {0} rejected: no upload handler configured", uploadId);
            this.notifyError(uploadId, null, "No upload handler configured");
            return;
        }
        try {
            fileData = Base64.getDecoder().decode(base64Data);
        }
        catch (IllegalArgumentException e) {
            logger.log(Level.WARNING, "Upload {0} rejected: invalid base64 data - {1}", new Object[]{uploadId, e.getMessage()});
            this.notifyError(uploadId, null, "Invalid file data: " + e.getMessage());
            return;
        }
        long fileSize = fileData.length;
        logger.log(Level.FINE, "Upload {0} decoded: {1} bytes", new Object[]{uploadId, fileSize});
        UploadHandler.UploadContext context = new UploadHandler.UploadContext(fileName, mimeType, fileSize);
        String validationError = this.uploadConfig.validate(context);
        if (validationError != null) {
            logger.log(Level.INFO, "Upload {0} rejected by validation: {1}", new Object[]{uploadId, validationError});
            this.notifyError(uploadId, null, validationError);
            return;
        }
        UploadTask task = new UploadTask(uploadId, fileName, mimeType, fileSize);
        this.activeTasks.put(uploadId, task);
        task.setStatus(UploadStatus.IN_PROGRESS);
        logger.log(Level.FINE, "Upload {0} task created and tracking started", uploadId);
        try {
            logger.log(Level.FINE, "Upload {0} invoking handler", uploadId);
            future = this.uploadHandler.handleUpload(context, new ByteArrayInputStream(fileData));
            if (future == null) {
                task.setStatus(UploadStatus.FAILED);
                task.setErrorMessage("Upload handler returned null");
                this.activeTasks.remove(uploadId);
                logger.log(Level.WARNING, "Upload {0} failed: handler returned null future", uploadId);
                this.notifyError(uploadId, task, "Upload handler returned null");
                return;
            }
        }
        catch (Exception e) {
            task.setStatus(UploadStatus.FAILED);
            Object errorMsg = e.getMessage();
            if (errorMsg == null || ((String)errorMsg).isEmpty()) {
                errorMsg = e.getClass().getSimpleName() + " occurred during upload initialization";
            }
            task.setErrorMessage((String)errorMsg);
            this.activeTasks.remove(uploadId);
            logger.log(Level.WARNING, "Upload {0} failed: handler threw {1} - {2}", new Object[]{uploadId, e.getClass().getSimpleName(), errorMsg});
            this.notifyError(uploadId, task, (String)errorMsg);
            return;
        }
        CompletableFuture<UploadHandler.UploadResult> timedFuture = this.uploadTimeoutSeconds > 0L ? future.orTimeout(this.uploadTimeoutSeconds, TimeUnit.SECONDS) : future;
        task.setFuture(timedFuture);
        timedFuture.handle((result, ex) -> {
            UploadTask uploadTask = task;
            synchronized (uploadTask) {
                if (task.isNotified()) {
                    logger.log(Level.FINE, "Upload {0} already notified, skipping duplicate callback", uploadId);
                    this.activeTasks.remove(uploadId);
                    return null;
                }
                if (task.getStatus() == UploadStatus.CANCELLED) {
                    logger.log(Level.FINE, "Upload {0} was cancelled, ignoring result", uploadId);
                    this.activeTasks.remove(uploadId);
                    return null;
                }
                long elapsedMs = task.getElapsedTimeMs();
                if (ex != null) {
                    Object errorMsg;
                    Throwable cause = ex instanceof CompletionException && ex.getCause() != null ? ex.getCause() : ex;
                    task.setStatus(UploadStatus.FAILED);
                    if (cause instanceof TimeoutException) {
                        errorMsg = "Upload timed out after " + this.uploadTimeoutSeconds + " seconds";
                        logger.log(Level.WARNING, "Upload {0} timed out after {1}ms", new Object[]{uploadId, elapsedMs});
                    } else {
                        errorMsg = cause.getMessage();
                        if (errorMsg == null || ((String)errorMsg).isEmpty()) {
                            errorMsg = cause.getClass().getSimpleName() + " occurred during upload";
                        }
                        logger.log(Level.WARNING, "Upload {0} failed after {1}ms: {2}", new Object[]{uploadId, elapsedMs, errorMsg});
                    }
                    task.setErrorMessage((String)errorMsg);
                    this.notifyResult(uploadId, task, null, (String)errorMsg);
                } else if (result != null && result.isSuccess()) {
                    task.setStatus(UploadStatus.COMPLETED);
                    task.setResultUrl(result.getUrl());
                    logger.log(Level.INFO, "Upload {0} completed successfully in {1}ms, url={2}", new Object[]{uploadId, elapsedMs, result.getUrl()});
                    this.notifyResult(uploadId, task, result.getUrl(), null);
                } else {
                    task.setStatus(UploadStatus.FAILED);
                    String errorMsg = result != null ? result.getErrorMessage() : "Unknown upload error";
                    task.setErrorMessage(errorMsg);
                    logger.log(Level.WARNING, "Upload {0} failed after {1}ms: {2}", new Object[]{uploadId, elapsedMs, errorMsg});
                    this.notifyResult(uploadId, task, null, errorMsg);
                }
                this.activeTasks.remove(uploadId);
            }
            return null;
        });
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public boolean cancelUpload(String uploadId) {
        UploadTask task = this.activeTasks.get(uploadId);
        if (task == null) {
            logger.log(Level.FINE, "Cancel request for unknown upload: {0}", uploadId);
            return false;
        }
        UploadTask uploadTask = task;
        synchronized (uploadTask) {
            if (task.isNotified()) {
                logger.log(Level.FINE, "Upload {0} already notified, cancel ignored", uploadId);
                return false;
            }
            if (task.getStatus() == UploadStatus.IN_PROGRESS || task.getStatus() == UploadStatus.PENDING) {
                task.setStatus(UploadStatus.CANCELLED);
                task.setErrorMessage("Upload cancelled");
                CompletableFuture<?> future = task.getFuture();
                if (future != null) {
                    future.cancel(true);
                }
                logger.log(Level.FINE, "Upload {0} cancelled after {1}ms", new Object[]{uploadId, task.getElapsedTimeMs()});
                this.notifyResult(uploadId, task, null, "Upload cancelled");
                this.activeTasks.remove(uploadId);
                return true;
            }
            logger.log(Level.FINE, "Upload {0} cannot be cancelled in status {1}", new Object[]{uploadId, task.getStatus()});
        }
        return false;
    }

    public int getActiveUploadCount() {
        return this.activeTasks.size();
    }

    public boolean hasActiveUploads() {
        return !this.activeTasks.isEmpty();
    }

    public UploadTask getUploadTask(String uploadId) {
        return this.activeTasks.get(uploadId);
    }

    public void cleanup() {
        for (String uploadId : new ArrayList<String>(this.activeTasks.keySet())) {
            this.cancelUpload(uploadId);
        }
        this.activeTasks.clear();
    }

    private void notifyError(String uploadId, UploadTask task, String error) {
        logger.log(Level.WARNING, "Upload failed for {0}: {1}", new Object[]{uploadId, error});
        this.notifyResult(uploadId, task, null, error);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void notifyResult(String uploadId, UploadTask task, String url, String error) {
        if (task != null) {
            UploadTask uploadTask = task;
            synchronized (uploadTask) {
                if (task.isNotified()) {
                    logger.log(Level.FINE, "Skipping duplicate notification for upload {0}", uploadId);
                    return;
                }
                task.setNotified(true);
            }
        }
        if (this.resultCallback != null) {
            try {
                this.resultCallback.onComplete(uploadId, url, error);
            }
            catch (Exception e) {
                logger.log(Level.SEVERE, "Error in upload result callback for " + uploadId, e);
            }
        }
    }

    @FunctionalInterface
    public static interface UploadResultCallback {
        public void onComplete(String var1, String var2, String var3);
    }

    public static class UploadTask {
        private final String uploadId;
        private final String fileName;
        private final String mimeType;
        private final long fileSize;
        private final long startTime;
        private volatile UploadStatus status;
        private volatile String resultUrl;
        private volatile String errorMessage;
        private volatile boolean notified;
        private volatile CompletableFuture<?> future;

        UploadTask(String uploadId, String fileName, String mimeType, long fileSize) {
            this.uploadId = uploadId;
            this.fileName = fileName;
            this.mimeType = mimeType;
            this.fileSize = fileSize;
            this.startTime = System.currentTimeMillis();
            this.status = UploadStatus.PENDING;
            this.notified = false;
        }

        public String getUploadId() {
            return this.uploadId;
        }

        public String getFileName() {
            return this.fileName;
        }

        public String getMimeType() {
            return this.mimeType;
        }

        public long getFileSize() {
            return this.fileSize;
        }

        public long getStartTime() {
            return this.startTime;
        }

        public UploadStatus getStatus() {
            return this.status;
        }

        public String getResultUrl() {
            return this.resultUrl;
        }

        public String getErrorMessage() {
            return this.errorMessage;
        }

        public boolean isNotified() {
            return this.notified;
        }

        void setStatus(UploadStatus status) {
            this.status = status;
        }

        void setResultUrl(String url) {
            this.resultUrl = url;
        }

        void setErrorMessage(String error) {
            this.errorMessage = error;
        }

        void setNotified(boolean notified) {
            this.notified = notified;
        }

        void setFuture(CompletableFuture<?> future) {
            this.future = future;
        }

        CompletableFuture<?> getFuture() {
            return this.future;
        }

        public long getElapsedTimeMs() {
            return System.currentTimeMillis() - this.startTime;
        }
    }

    public static enum UploadStatus {
        PENDING,
        IN_PROGRESS,
        COMPLETED,
        FAILED,
        CANCELLED;

    }
}

