/**
 * Upload Adapter Module for VaadinCKEditor
 *
 * Provides a custom CKEditor upload adapter that sends files to the Vaadin backend
 * via @ClientCallable methods, enabling server-side file handling through UploadHandler.
 */

/**
 * Upload promise resolver interface
 */
export interface UploadResolver {
    resolve: (url: string) => void;
    reject: (error: Error) => void;
}

/**
 * Server communication interface for upload operations
 */
export interface UploadServer {
    handleFileUpload(uploadId: string, fileName: string, mimeType: string, base64Data: string): void;
}

/**
 * CKEditor file loader interface
 */
export interface FileLoader {
    file: Promise<File>;
}

/**
 * CKEditor upload adapter interface
 */
export interface UploadAdapter {
    upload: () => Promise<{ default: string }>;
    abort: () => void;
}

/**
 * Logger interface for debugging
 */
interface Logger {
    debug: (...args: unknown[]) => void;
    warn: (...args: unknown[]) => void;
}

/**
 * Default allowed MIME types for image uploads.
 * These are the standard image formats supported by browsers.
 */
const DEFAULT_ALLOWED_MIME_TYPES = new Set([
    'image/jpeg',
    'image/png',
    'image/gif',
    'image/webp',
    'image/svg+xml',
    'image/bmp',
    'image/tiff',
]);

/**
 * Upload Adapter Manager class.
 * Manages file uploads from CKEditor to the Vaadin backend.
 */
export class UploadAdapterManager {
    private pendingUploads: Map<string, UploadResolver> = new Map();
    private uploadIdCounter = 0;
    private editorId: string;
    private server: UploadServer | undefined;
    private logger: Logger;
    private maxFileSize: number = 10 * 1024 * 1024; // 默认 10MB
    private allowedMimeTypes: Set<string> = new Set(DEFAULT_ALLOWED_MIME_TYPES);

    constructor(editorId: string, logger: Logger) {
        this.editorId = editorId;
        this.logger = logger;
    }

    /**
     * Set the maximum allowed file size for uploads.
     * @param bytes - Maximum file size in bytes
     */
    setMaxFileSize(bytes: number): void {
        this.maxFileSize = bytes;
    }

    /**
     * Set the allowed MIME types for uploads.
     * @param mimeTypes - Array of allowed MIME type strings
     */
    setAllowedMimeTypes(mimeTypes: string[]): void {
        this.allowedMimeTypes = new Set(mimeTypes);
    }

    /**
     * Add additional MIME types to the allowed list.
     * @param mimeTypes - Array of MIME type strings to add
     */
    addAllowedMimeTypes(mimeTypes: string[]): void {
        for (const type of mimeTypes) {
            this.allowedMimeTypes.add(type);
        }
    }

    /**
     * Check if a MIME type is allowed for upload.
     * @param mimeType - The MIME type to check
     * @returns true if allowed, false otherwise
     */
    isMimeTypeAllowed(mimeType: string): boolean {
        // Allow all types if whitelist is empty (disabled)
        if (this.allowedMimeTypes.size === 0) {
            return true;
        }
        return this.allowedMimeTypes.has(mimeType);
    }

    /**
     * Set the server reference for upload operations.
     */
    setServer(server: UploadServer | undefined): void {
        this.server = server;
    }

    /**
     * Create an upload adapter factory for CKEditor configuration.
     * Returns a function that creates upload adapters for each file upload.
     */
    createUploadAdapterFactory(): (loader: FileLoader) => UploadAdapter {
        const manager = this;

        return (loader: FileLoader): UploadAdapter => {
            return {
                upload: async (): Promise<{ default: string }> => {
                    const file = await loader.file;
                    if (!file) {
                        throw new Error('No file provided');
                    }

                    // 文件大小校验
                    if (file.size > manager.maxFileSize) {
                        throw new Error(`File size ${file.size} exceeds maximum allowed ${manager.maxFileSize} bytes`);
                    }

                    // MIME 类型白名单校验
                    if (!manager.isMimeTypeAllowed(file.type)) {
                        throw new Error(`File type '${file.type}' is not allowed. Allowed types: ${Array.from(manager.allowedMimeTypes).join(', ')}`);
                    }

                    const uploadId = `upload-${manager.editorId}-${++manager.uploadIdCounter}`;
                    const base64Data = await manager.fileToBase64(file);

                    const uploadPromise = new Promise<string>((resolve, reject) => {
                        manager.pendingUploads.set(uploadId, { resolve, reject });
                    });

                    if (manager.server) {
                        manager.server.handleFileUpload(uploadId, file.name, file.type, base64Data);
                    } else {
                        manager.pendingUploads.delete(uploadId);
                        throw new Error('Server connection not available');
                    }

                    const url = await uploadPromise;
                    return { default: url };
                },
                abort: () => {
                    manager.logger.debug('Upload abort requested');
                }
            };
        };
    }

    /**
     * Resolve a pending upload from server callback.
     * @param uploadId - The upload ID returned from handleFileUpload
     * @param url - The URL of the uploaded file (null if error)
     * @param errorMessage - Error message if upload failed (null if success)
     */
    resolveUpload(uploadId: string, url: string | null, errorMessage: string | null): void {
        const resolver = this.pendingUploads.get(uploadId);
        if (!resolver) {
            this.logger.warn(`No pending upload found for ID: ${uploadId}`);
            return;
        }

        this.pendingUploads.delete(uploadId);

        if (url) {
            resolver.resolve(url);
        } else {
            resolver.reject(new Error(errorMessage || 'Upload failed'));
        }
    }

    /**
     * Clean up all pending uploads.
     * Called when the component is disconnected.
     */
    cleanup(): void {
        if (this.pendingUploads.size > 0) {
            const disconnectError = new Error('Component disconnected');
            this.pendingUploads.forEach(resolver => resolver.reject(disconnectError));
            this.pendingUploads.clear();
        }
    }

    /**
     * Convert a file to Base64 string.
     */
    private fileToBase64(file: File): Promise<string> {
        return new Promise<string>((resolve, reject) => {
            const reader = new FileReader();
            reader.onload = () => {
                const result = reader.result as string;
                // Remove data URL prefix (e.g., "data:image/png;base64,")
                const base64 = result.split(',')[1];
                resolve(base64);
            };
            reader.onerror = () => reject(new Error('Failed to read file'));
            reader.readAsDataURL(file);
        });
    }

    /**
     * Check if there are pending uploads.
     */
    hasPendingUploads(): boolean {
        return this.pendingUploads.size > 0;
    }
}
