package com.vaadin.componentfactory;

/*
 * #%L
 * EnhancedRichTextEditor for Vaadin 10
 * %%
 * Copyright (C) 2017 - 2019 Vaadin Ltd
 * %%
 * This program is available under Commercial Vaadin Add-On License 3.0
 * (CVALv3).
 *
 * See the file license.html distributed with this software for more
 * information about licensing.
 *
 * You should have received a copy of the CVALv3 along with this program.
 * If not, see <http://vaadin.com/license/cval-3>.
 * #L%
 */

import com.vaadin.componentfactory.toolbar.ToolbarSlot;
import com.vaadin.flow.component.*;
import com.vaadin.flow.component.button.Button;
import com.vaadin.flow.component.dependency.JavaScript;
import com.vaadin.flow.component.dependency.JsModule;
import com.vaadin.flow.component.dependency.NpmPackage;
import com.vaadin.flow.component.icon.Icon;
import com.vaadin.flow.data.binder.Binder;
import com.vaadin.flow.data.value.HasValueChangeMode;
import com.vaadin.flow.data.value.ValueChangeMode;
import com.vaadin.flow.function.SerializableConsumer;
import com.vaadin.flow.internal.JsonSerializer;
import elemental.json.JsonArray;
import elemental.json.JsonObject;
import elemental.json.impl.JreJsonArray;
import elemental.json.impl.JreJsonFactory;
import org.jsoup.safety.Safelist;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Map;
import java.util.Objects;

/**
 * Server-side component for the {@code <vcf-enhanced-rich-text-editor>}
 * component.
 *
 * @author Vaadin Ltd
 */
@Tag("vcf-enhanced-rich-text-editor")
@JsModule("./richTextEditorConnector-npm.js")
@JavaScript("./richTextEditorConnector.js")
@NpmPackage(value = "@vaadin/vaadin-license-checker", version = "^2.1.2")
public class EnhancedRichTextEditor
        extends GeneratedEnhancedRichTextEditor<EnhancedRichTextEditor, String>
        implements HasSize, HasValueChangeMode, InputNotifier, KeyNotifier,
        CompositionNotifier {

    private ValueChangeMode currentMode;
    private RichTextEditorI18n i18n;
    private Map<ToolbarButton, Boolean> toolbarButtonsVisibility;
    private Collection<Placeholder> placeholders;

    /**
     * Gets the internationalization object previously set for this component.
     * <p>
     * Note: updating the object content that is gotten from this method will
     * not update the lang on the component if not set back using
     * {@link EnhancedRichTextEditor#setI18n(RichTextEditorI18n)}
     *
     * @return the i18n object. It will be <code>null</code>, If the i18n
     *         properties weren't set.
     */
    public RichTextEditorI18n getI18n() {
        return i18n;
    }

    /**
     * Sets the internationalization properties for this component.
     *
     * @param i18n
     *            the internationalized properties, not <code>null</code>
     */
    public void setI18n(RichTextEditorI18n i18n) {
        Objects.requireNonNull(i18n,
                "The I18N properties object should not be null");
        this.i18n = i18n;
        runBeforeClientResponse(ui -> {
            if (i18n == this.i18n) {
                JsonObject i18nObject = (JsonObject) JsonSerializer
                        .toJson(this.i18n);
                for (String key : i18nObject.keys()) {
                    ui.getPage().executeJs(
                            "$0.set('i18n." + key + "', $1)", getElement(),
                            i18nObject.get(key));
                }
            }
        });
    }

    public Map<ToolbarButton, Boolean> getToolbarButtonsVisibility() {
        return toolbarButtonsVisibility;
    }

    /**
     * Set which toolbar buttons are visible.
     * 
     * @param toolbarButtonsVisibility
     *            Map of button and boolean value. Boolean value false
     *            associated with the button means that button will be hidden.
     */
    public void setToolbarButtonsVisibility(
            Map<ToolbarButton, Boolean> toolbarButtonsVisibility) {
        this.toolbarButtonsVisibility = toolbarButtonsVisibility;
        runBeforeClientResponse(ui -> {
            String str = toolbarButtonsVisibility.toString();
            str = str.replaceAll("=", ":");
            ui.getPage().executeJs("setToolbarButtons($0, $1)",
                    getElement(), str);
        });
    }

    void runBeforeClientResponse(SerializableConsumer<UI> command) {
        getElement().getNode().runWhenAttached(ui -> ui
                .beforeClientResponse(this, context -> command.accept(ui)));
    }

    /**
     * Constructs an empty {@code EnhancedRichTextEditor}.
     */
    public EnhancedRichTextEditor() {
        super("", "", false);
        setValueChangeMode(ValueChangeMode.ON_CHANGE);
    }

    /**
     * Constructs a {@code EnhancedRichTextEditor} with the initial value
     *
     * @param initialValue
     *            the initial value
     * @see #setValue(Object)
     */
    public EnhancedRichTextEditor(String initialValue) {
        this();
        setValue(initialValue);
    }

    /**
     * Constructs an empty {@code TextField} with a value change listener.
     *
     * @param listener
     *            the value change listener
     * @see #addValueChangeListener(com.vaadin.flow.component.HasValue.ValueChangeListener)
     */
    public EnhancedRichTextEditor(
            ValueChangeListener<? super ComponentValueChangeEvent<EnhancedRichTextEditor, String>> listener) {
        this();
        addValueChangeListener(listener);
    }

    /**
     * Constructs an empty {@code EnhancedRichTextEditor} with a value change
     * listener and an initial value.
     *
     * @param initialValue
     *            the initial value
     * @param listener
     *            the value change listener
     * @see #setValue(Object)
     * @see #addValueChangeListener(com.vaadin.flow.component.HasValue.ValueChangeListener)
     */
    public EnhancedRichTextEditor(String initialValue,
            ValueChangeListener<? super ComponentValueChangeEvent<EnhancedRichTextEditor, String>> listener) {
        this();
        setValue(initialValue);
        addValueChangeListener(listener);
    }

    /**
     * {@inheritDoc}
     * <p>
     * The default value is {@link ValueChangeMode#ON_CHANGE}.
     */
    @Override
    public ValueChangeMode getValueChangeMode() {
        return currentMode;
    }

    @Override
    public void setValueChangeMode(ValueChangeMode valueChangeMode) {
        currentMode = valueChangeMode;
        setSynchronizedEvent(
                ValueChangeMode.eventForMode(valueChangeMode, "value-changed"));
    }

    /**
     * Sets the value of this editor. Should be in
     * <a href="https://github.com/quilljs/delta">Delta</a> format. If the new
     * value is not equal to {@code getValue()}, fires a value change event.
     * Throws {@code NullPointerException}, if the value is null.
     * <p>
     * Automatically detects and converts old-format deltas (containing
     * tabs-cont, pre-tab, line-part blots) to the new embed-based tab format.
     * <p>
     * Note: {@link Binder} will take care of the {@code null} conversion when
     * integrates with the editor, as long as no new converter is defined.
     *
     * @param value
     *            the new value in Delta format, not {@code null}
     */
    @Override
    public void setValue(String value) {
        super.setValue(TabConverter.convertIfNeeded(value));
    }

    /**
     * Returns the current value of the text editor in
     * <a href="https://github.com/quilljs/delta">Delta</a> format. By default,
     * the empty editor will return an empty string.
     *
     * @return the current value.
     */
    @Override
    public String getValue() {
        return super.getValue();
    }

    /**
     * Value of the editor presented as HTML string.
     *
     * @return the sanitized {@code htmlValue} property from the webcomponent.
     */
    public String getHtmlValue() {
        // Using basic whitelist and adding img tag with data protocol enabled.
        return sanitize(getHtmlValueString());
    }

    /**
     * Sets whether whitespace indicators are shown in the editor.
     * When true, special characters are displayed: → (tab), ↵ (soft-break),
     * ¶ (paragraph), ⮐ (auto-wrap).
     *
     * @param show true to show whitespace indicators
     */
    public void setShowWhitespace(boolean show) {
        getElement().setProperty("showWhitespace", show);
    }

    /**
     * Returns whether whitespace indicators are currently shown.
     *
     * @return true if whitespace indicators are visible
     */
    public boolean isShowWhitespace() {
        return getElement().getProperty("showWhitespace", false);
    }

    String sanitize(String html) {
        String cleaned = org.jsoup.Jsoup.clean(html,
                org.jsoup.safety.Safelist.basic()
                        .addTags("img", "h1", "h2", "h3", "s", "span", "br")
                        .addAttributes("img", "align", "alt", "height", "src",
                                "title", "width")
                        .addAttributes("span", "class", "contenteditable")
                        .addAttributes(":all", "style")
                        .addProtocols("img", "src", "data"));

        // Post-sanitization: restrict span class values to known safe classes
        // and contenteditable to only "false", to prevent CSS injection and
        // unintended editing behavior in contexts where HTML output is reused.
        org.jsoup.nodes.Document doc = org.jsoup.Jsoup.parse(cleaned);
        java.util.Set<String> allowedClasses = java.util.Set.of(
                "ql-tab", "ql-soft-break", "ql-readonly", "ql-placeholder");
        for (org.jsoup.nodes.Element span : doc.select("span[class]")) {
            String[] classes = span.attr("class").split("\\s+");
            StringBuilder filtered = new StringBuilder();
            for (String cls : classes) {
                if (allowedClasses.contains(cls)) {
                    if (filtered.length() > 0) filtered.append(' ');
                    filtered.append(cls);
                }
            }
            if (filtered.length() > 0) {
                span.attr("class", filtered.toString());
            } else {
                span.removeAttr("class");
            }
        }
        for (org.jsoup.nodes.Element span : doc.select("span[contenteditable]")) {
            if (!"false".equals(span.attr("contenteditable"))) {
                span.removeAttr("contenteditable");
            }
        }
        return doc.body().html();
    }

    /**
     * Set placeholders shown in the Placeholder drop down menu.
     * 
     * @param placeholders
     *            Collection of Placeholder objects
     */
    public void setPlaceholders(Collection<Placeholder> placeholders) {
        Objects.requireNonNull(placeholders, "placeholders cannot be null");
        JreJsonFactory factory = new JreJsonFactory();
        JsonArray jsonArray = new JreJsonArray(factory);

        int index = 0;
        for (Placeholder placeholder : placeholders) {
            jsonArray.set(index++, placeholder.toJson());
        }

        this.placeholders = placeholders;
        getElement().setPropertyJson("placeholders", jsonArray);
    }

    @Synchronize(property = "placeholders", value = "placeholders-changed")
    public Collection<Placeholder> getPlaceholders() {
        ArrayList<Placeholder> placeholders = new ArrayList<>();
        JsonArray rawArray = (JsonArray) getElement()
                .getPropertyRaw("placeholders");

        if (rawArray == null) {
            return placeholders;
        }

        for (int i = 0; i < rawArray.length(); i++) {
            JsonObject obj = rawArray.getObject(i);
            try {
                Placeholder placeholder = new Placeholder(obj);
                placeholders.add(placeholder);

            } catch (IllegalArgumentException e) {
            }
        }

        return placeholders;
    }

    public void setPlaceholderAltAppearancePattern(String pattern) {
        getElement().setProperty("placeholderAltAppearancePattern", pattern);
    }

    @Synchronize(property = "placeholderAltAppearancePattern", value = "placeholder-alt-appearance-pattern-changed")
    public String getPlaceholderAltAppearancePattern() {
        return getElement().getProperty("placeholderAltAppearancePattern");
    }

    public void setPlacehoderAltAppearance(boolean altAppearance) {
        getElement().setProperty("placeholderAltAppearance", altAppearance);
    }

    @Synchronize(property = "placeholderAltAppearance", value = "placeholder-alt-appearance-changed")
    public boolean isPlacehoderAltAppearance() {
        return getElement().getProperty("placeholderAltAppearance", false);
    }

    /**
     * For internal use only. Return Placeholder from the master list matching
     * the given Placeholder by getText.
     * 
     * @param placeholder
     *            The Placeholder to be searched.
     * @return A Placeholder
     */
    protected Placeholder getPlaceholder(Placeholder placeholder) {
        Objects.requireNonNull(placeholder, "Placeholder cannot be null");
        Objects.requireNonNull(placeholders,
                "getPlaceholder cannot be called before placeholders are set");
        return placeholders.stream()
                .filter(p -> p.getText().equals(placeholder.getText()))
                .findFirst().orElse(null);
    }

    /**
     * Return the length of the content stripped as text.
     * 
     * @return The length of the text content.
     */
    public int getTextLength() {
        return org.jsoup.Jsoup.clean(getHtmlValueString(),Safelist.none()).length();    
    }

    /**
     * Add a text to the position. Text will be added if position
     * is within 0 .. length of the current value of the text area.
     * 
     * @param text Text to be inserted
     * @param position Position
     */
    public void addText(String text, int position) {
        Objects.requireNonNull(text, "Text can't be null");
        this.getHtmlValue();
        if (position >= 0 && position <= getTextLength()) {
            getElement().executeJs("$0._editor.insertText($1,$2)", getElement(),
                    position, text);
        }
    }

    /**
     * Add text to the caret position, when the focus is in the text area.
     * 
     * @param text Text to be inserted
     */
    public void addText(String text) {
        Objects.requireNonNull(text, "Text can't be null");
        getElement().executeJs(
                "if ($0._editor.getSelection()) $0._editor.insertText($0._editor.getSelection().index,$1)",
                getElement(), text);
    }

    /**
     * Add a custom button to the toolbar.<br>
     * This method does NOT apply any toolbar styling to the button, but will keep it "Vaadin native".
     *
     * @param button A custom button to be added, not null
     * @deprecated use {@link #addCustomToolbarComponents(Component...)} instead
     */
    @Deprecated
    public void addCustomButton(Button button) {
        Objects.requireNonNull(button, "Button can't be null");
        addCustomToolbarComponents(button);
    }

    /**
     * A convenience method to add multiple custom buttons at one call.<br>
     * This method does NOT apply any toolbar styling to the button, but will keep it "Vaadin native".
     *
     * @param buttons Custom buttons to be added.
     * @deprecated use {@link #addCustomToolbarComponents(Component...)} instead
     */
    @Deprecated
    public void addCustomButtons(Button ...buttons) {
        addCustomToolbarComponents(buttons);
    }

    /**
     * A convenience method to add multiple custom components at one call. Uses the
     * {@link ToolbarSlot#GROUP_CUSTOM}.
     *
     * @param components Custom components to be added.
     */
    public void addCustomToolbarComponents(Component... components) {
        addToolbarComponents(ToolbarSlot.GROUP_CUSTOM, components);
    }

    /**
     * A convenience method to add multiple custom components at one call. Uses the
     * {@link ToolbarSlot#GROUP_CUSTOM}. The index allows to define the position of the newly added components
     * relative to already existing ones.
     *
     * @param index index
     * @param components Custom components to be added.
     */
    public void addCustomToolbarComponentsAtIndex(int index, Component... components) {
        addToolbarComponentsAtIndex(ToolbarSlot.GROUP_CUSTOM, index, components);
    }

    /**
     * Adds the components to the toolbar slot. Appends the components to existing ones.
     *
     * @param toolbarSlot slot to add the components to
     * @param components Components to be added
     */
    public void addToolbarComponents(ToolbarSlot toolbarSlot, Component... components) {
        Objects.requireNonNull(components);
        for (Component component : components) {
            Objects.requireNonNull(component);
            SlotUtil.addComponent(this, toolbarSlot.getSlotName(), component);
        }
    }

    /**
     * Adds the components to the toolbar slot. Appends the components to existing ones. The index allows to
     * define the position of the newly added components relative to already existing ones.
     *
     * @param toolbarSlot slot to add the components to
     * @param components Components to be added
     */
    public void addToolbarComponentsAtIndex(ToolbarSlot toolbarSlot, int index, Component... components) {
        Objects.requireNonNull(components);
        for (Component component : components) {
            Objects.requireNonNull(component);
            SlotUtil.addComponentAtIndex(this, toolbarSlot.getSlotName(), component, index);
        }
    }

    /**
     * Returns a toolbar component with the given id from the toolbar slot. The component must have been
     * added using one of the {@code addToolbarComponents} methods beforehand.
     * @param toolbarSlot toolbar slot
     * @param id component id
     * @return component
     * @param <T> return type
     */
    @SuppressWarnings("unchecked")
    public <T extends Component> T getToolbarComponent(ToolbarSlot toolbarSlot, String id) {
        Objects.requireNonNull(id, "Id can't be null");
        return (T) SlotUtil.getComponent(this, toolbarSlot.getSlotName(), id);
    }

    /**
     * Remove the given component from the toolbar. The component must have been
     * added using one of the {@code addToolbarComponents} methods beforehand.
     *
     * @param id component id
     */
    public void removeToolbarComponent(ToolbarSlot toolbarSlot, String id) {
        Objects.requireNonNull(id, "Id can't be null");
        SlotUtil.removeComponent(this, toolbarSlot.getSlotName(), id);
    }

    /**
     * Remove a custom component from the toolbar. The component must have been
     * added using one of the {@code addToolbarComponents} methods beforehand.
     *
     * @param component The component to be removed.
     */
    public void removeToolbarComponent(ToolbarSlot toolbarSlot, Component component) {
        Objects.requireNonNull(component, "Button can't be null");
        SlotUtil.removeComponent(this, toolbarSlot.getSlotName(), component);
    }

    /**
     * Get the custom button using its id.
     *
     * @param id Id as a string
     * @return A button
     * @deprecated use {@link #getToolbarComponent(ToolbarSlot, String)} instead with the {@link ToolbarSlot#GROUP_CUSTOM}
     */
    @Deprecated
    public Button getCustomButton(String id) {
        Objects.requireNonNull(id, "Id can't be null");
        return SlotUtil.getButton(this, id);
    }

    /**
     * Remove the given button from the toolbar.
     *
     * @param id Id as a string.
     * @deprecated use {@link #removeToolbarComponent(ToolbarSlot, String)} instead with the {@link ToolbarSlot#GROUP_CUSTOM}
     */
    @Deprecated
    public void removeCustomButton(String id) {
        Objects.requireNonNull(id, "Id can't be null");
        SlotUtil.removeButton(this, id);
    }

    /**
     * Remove a custom button from the toolbar.
     *
     * @param button The button to be removed.
     * @deprecated use {@link #removeToolbarComponent(ToolbarSlot, Component)} instead with the {@link ToolbarSlot#GROUP_CUSTOM}
     */
    @Deprecated
    public void removeCustomButton(Button button) {
        Objects.requireNonNull(button, "Button can't be null");
        SlotUtil.removeButton(this, button);
    }
    
    /**
     * Adds a custom shortcut to a specific toolbar standard button.
     * 
     * @param toolbarButton The toolbar button to add the shortcut to.
     * @param keyCode The key code for the new shortcut.
     * @param shortKey True if modifier ctrl is part of the shortcut.
     * @param shiftKey True if modifier shift is part of the shortcut.
     * @param altKey True if modifier alt is part of the shortcut.
     */
    public void addStandardToolbarButtonShortcut(ToolbarButton toolbarButton, Number keyCode,
        Boolean shortKey, Boolean shiftKey, Boolean altKey) {
      getElement().executeJs("$0.addStandardButtonBinding($1, $2, $3, $4, $5)", getElement(),
          toolbarButton.getButtonName(), keyCode, shortKey, shiftKey, altKey);
    }
    
    /**
     * Adds a custom shortcut to focus the editor toolbar.
     * 
     * @param keyCode The key code for the new shortcut.
     * @param shortKey True if modifier ctrl is part of the shortcut.
     * @param shiftKey True if modifier shift is part of the shortcut.
     * @param altKey True if modifier alt is part of the shortcut.
     */
    public void addToobarFocusShortcut(Number keyCode, Boolean shortKey, Boolean shiftKey,
        Boolean altKey) {
      getElement().executeJs("$0.addToolbarFocusBinding($1, $2, $3, $4)", getElement(), keyCode,
          shortKey, shiftKey, altKey);
    }
    
    /**
     * Allows to replace the icon of a standard {@link ToolbarButton toolbar button}.
     * 
     * @param toolbarButton toolbar button to replace icon
     * @param icon replacement icon
     */
    public void replaceStandardToolbarButtonIcon(ToolbarButton toolbarButton, Icon icon) {
   	 	Objects.requireNonNull(icon, "Icon can't be null");
        SlotUtil.replaceStandardButtonIcon(this, icon, toolbarButton.getButtonName());
    }

    /**
     * The internationalization properties for {@link EnhancedRichTextEditor}.
     */
    public static class RichTextEditorI18n implements Serializable {
        private String undo;
        private String redo;
        private String bold;
        private String italic;
        private String underline;
        private String strike;
        private String h1;
        private String h2;
        private String h3;
        private String subscript;
        private String superscript;
        private String listOrdered;
        private String listBullet;
        private String deindent;
        private String indent;
        private String alignLeft;
        private String alignCenter;
        private String alignRight;
        private String alignJustify;
        private String image;
        private String link;
        private String blockquote;
        private String codeBlock;
        private String readonly;
        private String placeholder;
        private String placeholderAppearance;
        private String placeholderComboBoxLabel;
        private String placeholderAppearanceLabel1;
        private String placeholderAppearanceLabel2;
        private String placeholderDialogTitle;
        private String clean;

        /**
         * Gets the translated word for {@code undo}
         *
         * @return the translated word for undo
         */
        public String getUndo() {
            return undo;
        }

        /**
         * Sets the translated word for {@code undo}.
         *
         * @param undo
         *            the translated word for undo
         * @return this instance for method chaining
         */
        public RichTextEditorI18n setUndo(String undo) {
            this.undo = undo;
            return this;
        }

        /**
         * Gets the translated word for {@code redo}
         *
         * @return the translated word for redo
         */
        public String getRedo() {
            return redo;
        }

        /**
         * Sets the translated word for {@code redo}.
         *
         * @param redo
         *            the translated word for redo
         * @return this instance for method chaining
         */
        public RichTextEditorI18n setRedo(String redo) {
            this.redo = redo;
            return this;
        }

        /**
         * Gets the translated word for {@code bold}
         *
         * @return the translated word for bold
         */
        public String getBold() {
            return bold;
        }

        /**
         * Sets the translated word for {@code bold}.
         *
         * @param bold
         *            the translated word for bold
         * @return this instance for method chaining
         */
        public RichTextEditorI18n setBold(String bold) {
            this.bold = bold;
            return this;
        }

        /**
         * Gets the translated word for {@code italic}
         *
         * @return the translated word for italic
         */
        public String getItalic() {
            return italic;
        }

        /**
         * Sets the translated word for {@code italic}.
         *
         * @param italic
         *            the translated word for italic
         * @return this instance for method chaining
         */
        public RichTextEditorI18n setItalic(String italic) {
            this.italic = italic;
            return this;
        }

        /**
         * Gets the translated word for {@code underline}
         *
         * @return the translated word for underline
         */
        public String getUnderline() {
            return underline;
        }

        /**
         * Sets the translated word for {@code underline}.
         *
         * @param underline
         *            the translated word for underline
         * @return this instance for method chaining
         */
        public RichTextEditorI18n setUnderline(String underline) {
            this.underline = underline;
            return this;
        }

        /**
         * Gets the translated word for {@code strike}
         *
         * @return the translated word for strike
         */
        public String getStrike() {
            return strike;
        }

        /**
         * Sets the translated word for {@code strike}.
         *
         * @param strike
         *            the translated word for strike
         * @return this instance for method chaining
         */
        public RichTextEditorI18n setStrike(String strike) {
            this.strike = strike;
            return this;
        }

        /**
         * Gets the translated word for {@code h1}
         *
         * @return the translated word for h1
         */
        public String getH1() {
            return h1;
        }

        /**
         * Sets the translated word for {@code h1}.
         *
         * @param h1
         *            the translated word for h1
         * @return this instance for method chaining
         */
        public RichTextEditorI18n setH1(String h1) {
            this.h1 = h1;
            return this;
        }

        /**
         * Gets the translated word for {@code h2}
         *
         * @return the translated word for h2
         */
        public String getH2() {
            return h2;
        }

        /**
         * Sets the translated word for {@code h2}.
         *
         * @param h2
         *            the translated word for h2
         * @return this instance for method chaining
         */
        public RichTextEditorI18n setH2(String h2) {
            this.h2 = h2;
            return this;
        }

        /**
         * Gets the translated word for {@code h3}
         *
         * @return the translated word for h3
         */
        public String getH3() {
            return h3;
        }

        /**
         * Sets the translated word for {@code h3}.
         *
         * @param h3
         *            the translated word for h3
         * @return this instance for method chaining
         */
        public RichTextEditorI18n setH3(String h3) {
            this.h3 = h3;
            return this;
        }

        /**
         * Gets the translated word for {@code subscript}
         *
         * @return the translated word for subscript
         */
        public String getSubscript() {
            return subscript;
        }

        /**
         * Sets the translated word for {@code subscript}.
         *
         * @param subscript
         *            the translated word for subscript
         * @return this instance for method chaining
         */
        public RichTextEditorI18n setSubscript(String subscript) {
            this.subscript = subscript;
            return this;
        }

        /**
         * Gets the translated word for {@code superscript}
         *
         * @return the translated word for superscript
         */
        public String getSuperscript() {
            return superscript;
        }

        /**
         * Sets the translated word for {@code superscript}.
         *
         * @param superscript
         *            the translated word for superscript
         * @return this instance for method chaining
         */
        public RichTextEditorI18n setSuperscript(String superscript) {
            this.superscript = superscript;
            return this;
        }

        /**
         * Gets the translated word for {@code listOrdered}
         *
         * @return the translated word for listOrdered
         */
        public String getListOrdered() {
            return listOrdered;
        }

        /**
         * Sets the translated word for {@code listOrdered}.
         *
         * @param listOrdered
         *            the translated word for listOrdered
         * @return this instance for method chaining
         */
        public RichTextEditorI18n setListOrdered(String listOrdered) {
            this.listOrdered = listOrdered;
            return this;
        }

        /**
         * Gets the translated word for {@code listBullet}
         *
         * @return the translated word for listBullet
         */
        public String getListBullet() {
            return listBullet;
        }

        /**
         * Sets the translated word for {@code listBullet}.
         *
         * @param listBullet
         *            the translated word for listBullet
         * @return this instance for method chaining
         */
        public RichTextEditorI18n setListBullet(String listBullet) {
            this.listBullet = listBullet;
            return this;
        }
        
        /**
         * Gets the translated word for {@code deindent}
         *
         * @return the translated word for deindent (outdent)
         */
        public String getDeindent() {
            return deindent;
        }

        /**
         * Sets the translated word for {@code deindent}.
         *
         * @param deindent
         *            the translated word for deindent
         * @return this instance for method chaining
         */
        public RichTextEditorI18n setDeindent(String deindent) {
            this.deindent = deindent;
            return this;
        }
        
        /**
         * Gets the translated word for {@code indent}
         *
         * @return the translated word for indent
         */
        public String getIndent() {
            return indent;
        }

        /**
         * Sets the translated word for {@code indent}.
         *
         * @param indent
         *            the translated word for indent
         * @return this instance for method chaining
         */
        public RichTextEditorI18n setIndent(String indent) {
            this.indent = indent;
            return this;
        }

        /**
         * Gets the translated word for {@code alignLeft}
         *
         * @return the translated word for alignLeft
         */
        public String getAlignLeft() {
            return alignLeft;
        }

        /**
         * Sets the translated word for {@code alignLeft}.
         *
         * @param alignLeft
         *            the translated word for alignLeft
         * @return this instance for method chaining
         */
        public RichTextEditorI18n setAlignLeft(String alignLeft) {
            this.alignLeft = alignLeft;
            return this;
        }

        /**
         * Gets the translated word for {@code alignCenter}
         *
         * @return the translated word for alignCenter
         */
        public String getAlignCenter() {
            return alignCenter;
        }

        /**
         * Sets the translated word for {@code alignCenter}.
         *
         * @param alignCenter
         *            the translated word for alignCenter
         * @return this instance for method chaining
         */
        public RichTextEditorI18n setAlignCenter(String alignCenter) {
            this.alignCenter = alignCenter;
            return this;
        }

        /**
         * Gets the translated word for {@code alignRight}
         *
         * @return the translated word for alignRight
         */
        public String getAlignRight() {
            return alignRight;
        }

        /**
         * Sets the translated word for {@code alignRight}.
         *
         * @param alignRight
         *            the translated word for alignRight
         * @return this instance for method chaining
         */
        public RichTextEditorI18n setAlignRight(String alignRight) {
            this.alignRight = alignRight;
            return this;
        }
        
        /**
         * Gets the translated word for {@code alignJustify}
         *
         * @return the translated word for alignJustify
         */
        public String getAlignJustify() {
            return alignJustify;
        }

        /**
         * Sets the translated word for {@code alignJustify}.
         *
         * @param alignJustify
         *            the translated word for alignJustify
         * @return this instance for method chaining
         */
        public RichTextEditorI18n setAlignJustify(String alignJustify) {
            this.alignJustify = alignJustify;
            return this;
        }
        

        /**
         * Gets the translated word for {@code image}
         *
         * @return the translated word for image
         */
        public String getImage() {
            return image;
        }

        /**
         * Sets the translated word for {@code image}.
         *
         * @param image
         *            the translated word for image
         * @return this instance for method chaining
         */
        public RichTextEditorI18n setImage(String image) {
            this.image = image;
            return this;
        }

        /**
         * Gets the translated word for {@code link}
         *
         * @return the translated word for link
         */
        public String getLink() {
            return link;
        }

        /**
         * Sets the translated word for {@code link}.
         *
         * @param link
         *            the translated word for link
         * @return this instance for method chaining
         */
        public RichTextEditorI18n setLink(String link) {
            this.link = link;
            return this;
        }

        /**
         * Gets the translated word for {@code blockquote}
         *
         * @return the translated word for blockquote
         */
        public String getBlockquote() {
            return blockquote;
        }

        /**
         * Sets the translated word for {@code blockquote}.
         *
         * @param blockquote
         *            the translated word for blockquote
         * @return this instance for method chaining
         */
        public RichTextEditorI18n setBlockquote(String blockquote) {
            this.blockquote = blockquote;
            return this;
        }

        /**
         * Gets the translated word for {@code codeBlock}
         *
         * @return the translated word for codeBlock
         */
        public String getCodeBlock() {
            return codeBlock;
        }

        /**
         * Sets the translated word for {@code codeBlock}.
         *
         * @param codeBlock
         *            the translated word for codeBlock
         * @return this instance for method chaining
         */
        public RichTextEditorI18n setCodeBlock(String codeBlock) {
            this.codeBlock = codeBlock;
            return this;
        }

        /**
         * Gets the translated word for {@code readonly}
         *
         * @return the translated word for readonly
         */
        public String getReadonly() {
            return readonly;
        }

        /**
         * Sets the translated word for {@code readonly}.
         *
         * @param readonly
         *            the translated word for readonly
         * @return this instance for method chaining
         */
        public RichTextEditorI18n setReadonly(String readonly) {
            this.readonly = readonly;
            return this;
        }

        /**
         * Gets the translated word for {@code placeholder}
         *
         * @return the translated word for placeholder
         */
        public String getPlaceholder() {
            return placeholder;
        }

        /**
         * Sets the translated word for {@code placeholder}.
         *
         * @param placeholder
         *            the translated word for placeholder
         * @return this instance for method chaining
         */
        public RichTextEditorI18n setPlaceholder(String placeholder) {
            this.placeholder = placeholder;
            return this;
        }

        /**
         * Gets the translated word for {@code placeholderAppearance}
         *
         * @return the translated word for placeholderAppearance
         */
        public String getPlaceholderAppearance() {
            return placeholderAppearance;
        }

        /**
         * Sets the translated word for {@code placeholderAppearance}.
         *
         * @param placeholderAppearance
         *            the translated word for placeholderAppearance
         * @return this instance for method chaining
         */
        public RichTextEditorI18n setPlaceholderAppeance(
                String placeholderAppearance) {
            this.placeholderAppearance = placeholderAppearance;
            return this;
        }

        /**
         * Gets the translated word for {@code placeholderComboBoxLabel}
         *
         * @return the translated word for placeholderComboBoxLabel
         */
        public String getPlaceholderComboBoxLabel() {
            return placeholderComboBoxLabel;
        }

        /**
         * Sets the translated word for {@code placeholderComboBoxLabel}.
         *
         * @param placeholderComboBoxLabel
         *            the translated word for placeholderComboBoxLabel
         * @return this instance for method chaining
         */
        public RichTextEditorI18n setPlaceholderComboBoxLabel(
                String placeholderComboBoxLabel) {
            this.placeholderComboBoxLabel = placeholderComboBoxLabel;
            return this;
        }

        /**
         * Gets the translated word for {@code placeholderAppearanceLabel1}
         *
         * @return the translated word for placeholderAppearanceLabel1
         */
        public String setPlaceholderAppearanceLabel1() {
            return placeholderAppearanceLabel1;
        }

        /**
         * Sets the translated word for {@code placeholderAppearanceLabel1}.
         *
         * @param placeholderAppearanceLabel1
         *            the translated word for placeholderAppearanceLabel1
         * @return this instance for method chaining
         */
        public RichTextEditorI18n getPlaceholderAppearanceLabel1(
                String placeholderAppearanceLabel1) {
            this.placeholderAppearanceLabel1 = placeholderAppearanceLabel1;
            return this;
        }

        /**
         * Gets the translated word for {@code placeholderAppearanceLabel2}
         *
         * @return the translated word for placeholderAppearanceLabel2
         */
        public String setPlaceholderAppearanceLabel2() {
            return placeholderAppearanceLabel2;
        }

        /**
         * Sets the translated word for {@code placeholderAppearanceLabel2}.
         *
         * @param placeholderAppearanceLabel2
         *            the translated word for placeholderAppearanceLabel2
         * @return this instance for method chaining
         */
        public RichTextEditorI18n getPlaceholderAppearanceLabel2(
                String placeholderAppearanceLabel2) {
            this.placeholderAppearanceLabel2 = placeholderAppearanceLabel2;
            return this;
        }

        /**
         * Gets the translated word for {@code placeholderDialogTitle}
         *
         * @return the translated word for placeholderDialogTitle
         */
        public String getPlaceholderDialogTitle() {
            return placeholderDialogTitle;
        }

        /**
         * Sets the translated word for {@code placeholderDialogTitle}.
         *
         * @param placeholderDialogTitle
         *            the translated word for placeholderDialogTitle
         * @return this instance for method chaining
         */
        public RichTextEditorI18n getPlaceholderDialogTitle(
                String placeholderDialogTitle) {
            this.placeholderDialogTitle = placeholderDialogTitle;
            return this;
        }

        /**
         * Gets the translated word for {@code clean}
         *
         * @return the translated word for clean
         */
        public String getClean() {
            return clean;
        }

        /**
         * Sets the translated word for {@code clean}.
         *
         * @param clean
         *            the translated word for clean
         * @return this instance for method chaining
         */
        public RichTextEditorI18n setClean(String clean) {
            this.clean = clean;
            return this;
        }

        /**
         * Gets the stringified values of the tooltips.
         *
         * @return stringified values of the tooltips
         */
        @Override
        public String toString() {
            return "[" + undo + ", " + redo + ", " + bold + ", " + italic + ", "
                   + underline + ", " + strike + ", " + h1 + ", " + h2 + ", "
                   + h3 + ", " + subscript + ", " + superscript + ", "
                   + listOrdered + ", " + listBullet + ", " + deindent
                   + ", " + indent + ", " + alignLeft + ", " + alignCenter
                   + ", " + alignRight + ", " + alignJustify + ", " + image + ", "
                   + link + ", " + blockquote + ", " + codeBlock + ", "
                   + readonly + ", " + placeholder + ", "
                   + placeholderAppearance + ", " + placeholderComboBoxLabel
                   + ", " + placeholderAppearanceLabel1 + ", "
                   + placeholderAppearanceLabel2 + ", "
                   + placeholderDialogTitle + ", " + clean + "]";
        }
    }

    public enum ToolbarButton {
        UNDO, REDO, BOLD, ITALIC, UNDERLINE, STRIKE, H1, H2, H3, SUBSCRIPT, SUPERSCRIPT, LIST_ORDERED, LIST_BULLET, DEINDENT, INDENT, ALIGN_LEFT, ALIGN_CENTER, ALIGN_RIGHT, ALIGN_JUSTIFY, IMAGE, LINK, BLOCKQUOTE, CODE_BLOCK, WHITESPACE, READONLY, CLEAN, PLACEHOLDER, PLACEHOLDER_APPEARANCE;

        @Override
        public String toString() {
            String name = getButtonName();
            return "\"" + name + "\"";
        }
        
        public String getButtonName() {
          String str = this.name().toLowerCase();
          String[] parts = str.split("_");
          if (parts.length == 1)
              return str;

          for (int i = 1; i < parts.length; i++)
              parts[i] = Character.toUpperCase(parts[i].charAt(0))
                      + parts[i].substring(1);
          
          return String.join("", parts);
        }
        
    }

}
